/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.cache;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import org.springframework.dao.PessimisticLockingFailureException;
import org.springframework.data.redis.cache.BatchStrategy;
import org.springframework.data.redis.cache.CacheStatistics;
import org.springframework.data.redis.cache.CacheStatisticsCollector;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.types.Expiration;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;

class DefaultRedisCacheWriter
implements RedisCacheWriter {
    private final RedisConnectionFactory connectionFactory;
    private final Duration sleepTime;
    private final CacheStatisticsCollector statistics;
    private final BatchStrategy batchStrategy;

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, BatchStrategy batchStrategy) {
        this(connectionFactory, Duration.ZERO, batchStrategy);
    }

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, BatchStrategy batchStrategy) {
        this(connectionFactory, sleepTime, CacheStatisticsCollector.none(), batchStrategy);
    }

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, CacheStatisticsCollector cacheStatisticsCollector, BatchStrategy batchStrategy) {
        Assert.notNull((Object)connectionFactory, "ConnectionFactory must not be null!");
        Assert.notNull((Object)sleepTime, "SleepTime must not be null!");
        Assert.notNull((Object)cacheStatisticsCollector, "CacheStatisticsCollector must not be null!");
        Assert.notNull((Object)batchStrategy, "BatchStrategy must not be null!");
        this.connectionFactory = connectionFactory;
        this.sleepTime = sleepTime;
        this.statistics = cacheStatisticsCollector;
        this.batchStrategy = batchStrategy;
    }

    @Override
    public void put(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, "Name must not be null!");
        Assert.notNull((Object)key, "Key must not be null!");
        Assert.notNull((Object)value, "Value must not be null!");
        this.execute(name, connection -> {
            if (DefaultRedisCacheWriter.shouldExpireWithin(ttl)) {
                connection.set(key, value, Expiration.from(ttl.toMillis(), TimeUnit.MILLISECONDS), RedisStringCommands.SetOption.upsert());
            } else {
                connection.set(key, value);
            }
            return "OK";
        });
        this.statistics.incPuts(name);
    }

    @Override
    public byte[] get(String name, byte[] key) {
        Assert.notNull((Object)name, "Name must not be null!");
        Assert.notNull((Object)key, "Key must not be null!");
        byte[] result = this.execute(name, connection -> connection.get(key));
        this.statistics.incGets(name);
        if (result != null) {
            this.statistics.incHits(name);
        } else {
            this.statistics.incMisses(name);
        }
        return result;
    }

    @Override
    public byte[] putIfAbsent(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, "Name must not be null!");
        Assert.notNull((Object)key, "Key must not be null!");
        Assert.notNull((Object)value, "Value must not be null!");
        return this.execute(name, connection -> {
            if (this.isLockingCacheWriter()) {
                this.doLock(name, (RedisConnection)connection);
            }
            try {
                boolean put = DefaultRedisCacheWriter.shouldExpireWithin(ttl) ? connection.set(key, value, Expiration.from(ttl), RedisStringCommands.SetOption.ifAbsent()).booleanValue() : connection.setNX(key, value).booleanValue();
                if (put) {
                    this.statistics.incPuts(name);
                    byte[] byArray = null;
                    return byArray;
                }
                byte[] byArray = connection.get(key);
                return byArray;
            }
            finally {
                if (this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
        });
    }

    @Override
    public void remove(String name, byte[] key) {
        Assert.notNull((Object)name, "Name must not be null!");
        Assert.notNull((Object)key, "Key must not be null!");
        this.execute(name, connection -> connection.del(new byte[][]{key}));
        this.statistics.incDeletes(name);
    }

    @Override
    public void clean(String name, byte[] pattern) {
        Assert.notNull((Object)name, "Name must not be null!");
        Assert.notNull((Object)pattern, "Pattern must not be null!");
        this.execute(name, connection -> {
            boolean wasLocked = false;
            try {
                long deleteCount;
                if (this.isLockingCacheWriter()) {
                    this.doLock(name, (RedisConnection)connection);
                    wasLocked = true;
                }
                for (deleteCount = this.batchStrategy.cleanCache((RedisConnection)connection, name, pattern); deleteCount > Integer.MAX_VALUE; deleteCount -= Integer.MAX_VALUE) {
                    this.statistics.incDeletesBy(name, Integer.MAX_VALUE);
                }
                this.statistics.incDeletesBy(name, (int)deleteCount);
            }
            finally {
                if (wasLocked && this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
            return "OK";
        });
    }

    @Override
    public CacheStatistics getCacheStatistics(String cacheName) {
        return this.statistics.getCacheStatistics(cacheName);
    }

    @Override
    public void clearStatistics(String name) {
        this.statistics.reset(name);
    }

    @Override
    public RedisCacheWriter withStatisticsCollector(CacheStatisticsCollector cacheStatisticsCollector) {
        return new DefaultRedisCacheWriter(this.connectionFactory, this.sleepTime, cacheStatisticsCollector, this.batchStrategy);
    }

    void lock(String name) {
        this.execute(name, connection -> this.doLock(name, (RedisConnection)connection));
    }

    void unlock(String name) {
        this.executeLockFree(connection -> this.doUnlock(name, (RedisConnection)connection));
    }

    private Boolean doLock(String name, RedisConnection connection) {
        return connection.setNX(DefaultRedisCacheWriter.createCacheLockKey(name), new byte[0]);
    }

    private Long doUnlock(String name, RedisConnection connection) {
        return connection.del(new byte[][]{DefaultRedisCacheWriter.createCacheLockKey(name)});
    }

    boolean doCheckLock(String name, RedisConnection connection) {
        return connection.exists(DefaultRedisCacheWriter.createCacheLockKey(name));
    }

    private boolean isLockingCacheWriter() {
        return !this.sleepTime.isZero() && !this.sleepTime.isNegative();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T execute(String name, Function<RedisConnection, T> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            this.checkAndPotentiallyWaitUntilUnlocked(name, connection);
            T t = callback.apply(connection);
            return t;
        }
    }

    private void executeLockFree(Consumer<RedisConnection> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            callback.accept(connection);
        }
    }

    private void checkAndPotentiallyWaitUntilUnlocked(String name, RedisConnection connection) {
        if (!this.isLockingCacheWriter()) {
            return;
        }
        long lockWaitTimeNs = System.nanoTime();
        try {
            while (this.doCheckLock(name, connection)) {
                Thread.sleep(this.sleepTime.toMillis());
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new PessimisticLockingFailureException(String.format("Interrupted while waiting to unlock cache %s", name), ex);
        }
        finally {
            this.statistics.incLockTime(name, System.nanoTime() - lockWaitTimeNs);
        }
    }

    private static boolean shouldExpireWithin(@Nullable Duration ttl) {
        return ttl != null && !ttl.isZero() && !ttl.isNegative();
    }

    private static byte[] createCacheLockKey(String name) {
        return (name + "~lock").getBytes(StandardCharsets.UTF_8);
    }
}

