/*
 * Licensed to ElasticSearch and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. ElasticSearch licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.lucene.index;

import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.metrics.MeanMetric;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;

import java.io.IOException;

// LUCENE MONITOR - Copied from SerialMergeScheduler
public class TrackingSerialMergeScheduler extends MergeScheduler {

    protected final ESLogger logger;

    private final MeanMetric totalMerges = new MeanMetric();
    private final CounterMetric totalMergesNumDocs = new CounterMetric();
    private final CounterMetric totalMergesSizeInBytes = new CounterMetric();
    private final CounterMetric currentMerges = new CounterMetric();
    private final CounterMetric currentMergesNumDocs = new CounterMetric();
    private final CounterMetric currentMergesSizeInBytes = new CounterMetric();

    public TrackingSerialMergeScheduler(ESLogger logger) {
        this.logger = logger;
    }

    public long totalMerges() {
        return totalMerges.count();
    }

    public long totalMergeTime() {
        return totalMerges.sum();
    }

    public long totalMergeNumDocs() {
        return totalMergesNumDocs.count();
    }

    public long totalMergeSizeInBytes() {
        return totalMergesSizeInBytes.count();
    }

    public long currentMerges() {
        return currentMerges.count();
    }

    public long currentMergesNumDocs() {
        return currentMergesNumDocs.count();
    }

    public long currentMergesSizeInBytes() {
        return currentMergesSizeInBytes.count();
    }

    /**
     * Just do the merges in sequence. We do this
     * "synchronized" so that even if the application is using
     * multiple threads, only one merge may run at a time.
     */
    @Override
    synchronized public void merge(IndexWriter writer) throws CorruptIndexException, IOException {
        while (true) {
            MergePolicy.OneMerge merge = writer.getNextMerge();
            if (merge == null)
                break;

            // different from serial merge, call mergeInit here so we get the correct stats
            // mergeInit can be called several times without side affects (checks on merge.info not being null)
            writer.mergeInit(merge);

            int totalNumDocs = merge.totalNumDocs();
            long totalSizeInBytes = merge.estimatedMergeBytes;
            long time = System.currentTimeMillis();
            currentMerges.inc();
            currentMergesNumDocs.inc(totalNumDocs);
            currentMergesSizeInBytes.inc(totalSizeInBytes);

            // sadly, segment name is not available since mergeInit is called from merge itself...
            if (logger.isTraceEnabled()) {
                logger.trace("merge [{}] starting..., merging [{}] segments, [{}] docs, [{}] size, into [{}] estimated_size", merge.info == null ? "_na_" : merge.info.name, merge.segments.size(), totalNumDocs, new ByteSizeValue(totalSizeInBytes), new ByteSizeValue(merge.estimatedMergeBytes));
            }
            try {
                TrackingMergeScheduler.setCurrentMerge(merge);
                writer.merge(merge);
            } finally {
                TrackingMergeScheduler.removeCurrentMerge();
                long took = System.currentTimeMillis() - time;

                currentMerges.dec();
                currentMergesNumDocs.dec(totalNumDocs);
                currentMergesSizeInBytes.dec(totalSizeInBytes);

                totalMergesNumDocs.inc(totalNumDocs);
                totalMergesSizeInBytes.inc(totalSizeInBytes);
                totalMerges.inc(took);
                if (took > 20000) { // if more than 20 seconds, DEBUG log it
                    logger.debug("merge [{}] done, took [{}]", merge.info == null ? "_na_" : merge.info.name, TimeValue.timeValueMillis(took));
                } else if (logger.isTraceEnabled()) {
                    logger.trace("merge [{}] done, took [{}]", merge.info == null ? "_na_" : merge.info.name, TimeValue.timeValueMillis(took));
                }
            }
        }
    }

    @Override
    public void close() {
    }
}