/*
 * Licensed to Elastic Search and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. Elastic Search licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.field.data.bytes;

import gnu.trove.list.array.TByteArrayList;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.FieldCache;
import org.elasticsearch.common.RamUsage;
import org.elasticsearch.index.field.data.FieldDataType;
import org.elasticsearch.index.field.data.NumericFieldData;
import org.elasticsearch.index.field.data.support.FieldDataLoader;

import java.io.IOException;

/**
 *
 */
public abstract class ByteFieldData extends NumericFieldData<ByteDocFieldData> {

    static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

    protected final byte[] values;

    protected ByteFieldData(String fieldName, byte[] values) {
        super(fieldName);
        this.values = values;
    }

    @Override
    protected long computeSizeInBytes() {
        return 1 * values.length + RamUsage.NUM_BYTES_ARRAY_HEADER;
    }

    public final byte[] values() {
        return this.values;
    }

    abstract public byte value(int docId);

    abstract public byte[] values(int docId);

    @Override
    public ByteDocFieldData docFieldData(int docId) {
        return super.docFieldData(docId);
    }

    @Override
    protected ByteDocFieldData createFieldData() {
        return new ByteDocFieldData(this);
    }

    @Override
    public void forEachValue(StringValueProc proc) {
        for (int i = 1; i < values.length; i++) {
            proc.onValue(Byte.toString(values[i]));
        }
    }

    @Override
    public String stringValue(int docId) {
        return Byte.toString(value(docId));
    }

    @Override
    public byte byteValue(int docId) {
        return value(docId);
    }

    @Override
    public short shortValue(int docId) {
        return value(docId);
    }

    @Override
    public int intValue(int docId) {
        return (int) value(docId);
    }

    @Override
    public long longValue(int docId) {
        return (long) value(docId);
    }

    @Override
    public float floatValue(int docId) {
        return (float) value(docId);
    }

    @Override
    public double doubleValue(int docId) {
        return (double) value(docId);
    }

    @Override
    public FieldDataType type() {
        return FieldDataType.DefaultTypes.BYTE;
    }

    public void forEachValue(ValueProc proc) {
        for (int i = 1; i < values.length; i++) {
            proc.onValue(values[i]);
        }
    }

    public static interface ValueProc {
        void onValue(byte value);
    }

    public abstract void forEachValueInDoc(int docId, ValueInDocProc proc);

    public static interface ValueInDocProc {
        void onValue(int docId, byte value);

        void onMissing(int docID);
    }

    public static ByteFieldData load(IndexReader reader, String field) throws IOException {
        return FieldDataLoader.load(reader, field, new ByteTypeLoader());
    }

    static class ByteTypeLoader extends FieldDataLoader.FreqsTypeLoader<ByteFieldData> {

        private final TByteArrayList terms = new TByteArrayList();

        ByteTypeLoader() {
            super();
            // the first one indicates null value
            terms.add((byte) 0);
        }

        @Override
        public void collectTerm(String term) {
            terms.add((byte) FieldCache.NUMERIC_UTILS_INT_PARSER.parseInt(term));
        }

        @Override
        public ByteFieldData buildSingleValue(String field, int[] ordinals) {
            return new SingleValueByteFieldData(field, ordinals, terms.toArray());
        }

        @Override
        public ByteFieldData buildMultiValue(String field, int[][] ordinals) {
            return new MultiValueByteFieldData(field, ordinals, terms.toArray());
        }
    }
}