/*
 * Licensed to ElasticSearch and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. ElasticSearch licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.field.data.shorts;

import org.elasticsearch.index.cache.field.data.FieldDataCache;
import org.elasticsearch.index.field.data.FieldDataType;
import org.elasticsearch.index.field.data.support.NumericFieldDataComparator;

/**
 *
 */
// LUCENE MONITOR: Monitor against FieldComparator.Short
public class ShortFieldDataMissingComparator extends NumericFieldDataComparator {

    private final short[] values;
    private short bottom;
    private final short missingValue;

    public ShortFieldDataMissingComparator(int numHits, String fieldName, FieldDataCache fieldDataCache, short missingValue) {
        super(fieldName, fieldDataCache);
        values = new short[numHits];
        this.missingValue = missingValue;
    }

    @Override
    public FieldDataType fieldDataType() {
        return FieldDataType.DefaultTypes.SHORT;
    }

    @Override
    public int compare(int slot1, int slot2) {
        return values[slot1] - values[slot2];
    }

    @Override
    public int compareBottom(int doc) {
        short value = missingValue;
        if (currentFieldData.hasValue(doc)) {
            value = currentFieldData.shortValue(doc);
        }
        return bottom - value;
    }

    @Override
    public void copy(int slot, int doc) {
        short value = missingValue;
        if (currentFieldData.hasValue(doc)) {
            value = currentFieldData.shortValue(doc);
        }
        values[slot] = value;
    }

    @Override
    public void setBottom(final int bottom) {
        this.bottom = values[bottom];
    }

    @Override
    public Comparable value(int slot) {
        return Short.valueOf(values[slot]);
    }
}
