/*
 * Licensed to ElasticSearch and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. ElasticSearch licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.query;

import com.google.common.collect.Lists;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.search.geo.GeoHashUtils;
import org.elasticsearch.index.search.geo.Point;

import java.io.IOException;
import java.util.List;

/**
 *
 */
public class GeoPolygonFilterBuilder extends BaseFilterBuilder {

    private final String name;

    private final List<Point> points = Lists.newArrayList();

    private Boolean cache;
    private String cacheKey;

    private String filterName;

    public GeoPolygonFilterBuilder(String name) {
        this.name = name;
    }

    /**
     * Adds a point with lat and lon
     *
     * @param lat The latitude
     * @param lon The longitude
     * @return
     */
    public GeoPolygonFilterBuilder addPoint(double lat, double lon) {
        points.add(new Point(lat, lon));
        return this;
    }

    public GeoPolygonFilterBuilder addPoint(String geohash) {
        double[] values = GeoHashUtils.decode(geohash);
        return addPoint(values[0], values[1]);
    }

    /**
     * Sets the filter name for the filter that can be used when searching for matched_filters per hit.
     */
    public GeoPolygonFilterBuilder filterName(String filterName) {
        this.filterName = filterName;
        return this;
    }

    /**
     * Should the filter be cached or not. Defaults to <tt>false</tt>.
     */
    public GeoPolygonFilterBuilder cache(boolean cache) {
        this.cache = cache;
        return this;
    }

    public GeoPolygonFilterBuilder cacheKey(String cacheKey) {
        this.cacheKey = cacheKey;
        return this;
    }

    @Override
    protected void doXContent(XContentBuilder builder, Params params) throws IOException {
        builder.startObject(GeoPolygonFilterParser.NAME);

        builder.startObject(name);
        builder.startArray("points");
        for (Point point : points) {
            builder.startArray().value(point.lon).value(point.lat).endArray();
        }
        builder.endArray();
        builder.endObject();

        if (filterName != null) {
            builder.field("_name", filterName);
        }
        if (cache != null) {
            builder.field("_cache", cache);
        }
        if (cacheKey != null) {
            builder.field("_cache_key", cacheKey);
        }

        builder.endObject();
    }
}
