/*
 * Licensed to Elastic Search and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. Elastic Search licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.search.child;

import gnu.trove.map.TObjectFloatMap;
import gnu.trove.map.TObjectIntMap;
import gnu.trove.map.hash.TObjectFloatHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.*;
import org.apache.lucene.util.ToStringUtils;
import org.elasticsearch.ElasticSearchIllegalStateException;
import org.elasticsearch.common.CacheRecycler;
import org.elasticsearch.common.bytes.HashedBytesArray;
import org.elasticsearch.index.cache.id.IdReaderTypeCache;
import org.elasticsearch.search.internal.ScopePhase;
import org.elasticsearch.search.internal.SearchContext;

import java.io.IOException;
import java.util.Set;

/**
 * A query implementation that executes the wrapped child query and
 * connects the matching child docs to the related parent documents
 * using the {@link IdReaderTypeCache}.
 */
public class ChildrenQuery extends Query implements ScopePhase.CollectorPhase {

    private final SearchContext searchContext;
    private final String parentType;
    private final String childType;
    private final Filter parentFilter;
    private final ScoreType scoreType;
    private final String scope;
    private final Query childQuery;

    private TObjectFloatHashMap<HashedBytesArray> uidToScore;
    private TObjectIntHashMap<HashedBytesArray> uidToCount;

    public ChildrenQuery(SearchContext searchContext, String parentType, String childType, Filter parentFilter, String scope, Query childQuery, ScoreType scoreType) {
        this.searchContext = searchContext;
        this.parentType = parentType;
        this.childType = childType;
        this.parentFilter = parentFilter;
        this.scope = scope;
        this.childQuery = childQuery;
        this.scoreType = scoreType;
    }

    private ChildrenQuery(ChildrenQuery unProcessedQuery, Query rewrittenChildQuery) {
        this.searchContext = unProcessedQuery.searchContext;
        this.parentType = unProcessedQuery.parentType;
        this.childType = unProcessedQuery.childType;
        this.parentFilter = unProcessedQuery.parentFilter;
        this.scope = unProcessedQuery.scope;
        this.scoreType = unProcessedQuery.scoreType;
        this.childQuery = rewrittenChildQuery;

        this.uidToScore = unProcessedQuery.uidToScore;
        this.uidToCount = unProcessedQuery.uidToCount;
    }

    @Override
    public String toString(String field) {
        StringBuilder sb = new StringBuilder();
        sb.append("ChildrenQuery[").append(childType).append("/").append(parentType).append("](").append(childQuery
                .toString(field)).append(')').append(ToStringUtils.boost(getBoost()));
        return sb.toString();
    }

    @Override
    public Query rewrite(IndexReader reader) throws IOException {
        Query rewrittenChildQuery = childQuery.rewrite(reader);
        if (rewrittenChildQuery == childQuery) {
            return this;
        }

        int index = searchContext.scopePhases().indexOf(this);
        ChildrenQuery rewrite = new ChildrenQuery(this, rewrittenChildQuery);
        searchContext.scopePhases().set(index, rewrite);
        return rewrite;
    }

    @Override
    public void extractTerms(Set<Term> terms) {
        childQuery.extractTerms(terms);
    }

    @Override
    public boolean requiresProcessing() {
        return uidToScore == null;
    }

    @Override
    public Collector collector() {
        uidToScore = CacheRecycler.popObjectFloatMap();
        switch (scoreType) {
            case AVG:
                uidToCount = CacheRecycler.popObjectIntMap();
                return new AvgChildUidCollector(scoreType, searchContext, parentType, uidToScore, uidToCount);
            default:
                return new ChildUidCollector(scoreType, searchContext, parentType, uidToScore);
        }
    }

    @Override
    public void processCollector(Collector collector) {
        // Do nothing, we already have the references to the child scores and optionally the child count.
    }

    @Override
    public String scope() {
        return scope;
    }

    @Override
    public void clear() {
        if (uidToScore != null) {
            CacheRecycler.pushObjectFloatMap(uidToScore);
        }
        uidToScore = null;
        if (uidToCount != null) {
            CacheRecycler.pushObjectIntMap(uidToCount);
        }
        uidToCount = null;
    }

    @Override
    public Query query() {
        return childQuery;
    }

    @Override
    public Weight createWeight(Searcher searcher) throws IOException {
        if (uidToScore == null) {
            throw new ElasticSearchIllegalStateException("has_child query hasn't executed properly");
        }

        return new ParentWeight(childQuery.createWeight(searcher));
    }

    class ParentWeight extends Weight {

        final Weight childWeight;

        public ParentWeight(Weight childWeight) {
            this.childWeight = childWeight;
        }

        @Override
        public Query getQuery() {
            return ChildrenQuery.this;
        }

        @Override
        public Explanation explain(IndexReader reader, int doc) throws IOException {
            return new Explanation(getBoost(), "not implemented yet...");
        }

        @Override
        public float getValue() {
            return getBoost();
        }

        @Override
        public void normalize(float norm) {
        }

        @Override
        public float sumOfSquaredWeights() throws IOException {
            float sum = childWeight.sumOfSquaredWeights();
            sum *= getBoost() * getBoost();
            return sum;
        }

        @Override
        public Scorer scorer(IndexReader reader, boolean scoreDocsInOrder, boolean topScorer) throws IOException {
            DocIdSet parentsSet = parentFilter.getDocIdSet(reader);
            if (parentsSet == null || parentsSet == DocIdSet.EMPTY_DOCIDSET) {
                return null;
            }

            IdReaderTypeCache idTypeCache = searchContext.idCache().reader(reader).type(parentType);
            DocIdSetIterator parentsIterator = parentsSet.iterator();
            switch (scoreType) {
                case AVG:
                    return new AvgParentScorer(this, idTypeCache, uidToScore, uidToCount, parentsIterator);
                default:
                    return new ParentScorer(this, idTypeCache, uidToScore, parentsIterator);
            }
        }

    }

    static class ParentScorer extends Scorer {

        final IdReaderTypeCache idTypeCache;
        final TObjectFloatMap<HashedBytesArray> uidToScore;
        final DocIdSetIterator parentsIterator;

        int currentDocId = -1;
        float currentScore;

        ParentScorer(Weight weight, IdReaderTypeCache idTypeCache, TObjectFloatMap<HashedBytesArray> uidToScore, DocIdSetIterator parentsIterator) {
            super(weight);
            this.idTypeCache = idTypeCache;
            this.uidToScore = uidToScore;
            this.parentsIterator = parentsIterator;
        }

        @Override
        public float score() throws IOException {
            return currentScore;
        }

        @Override
        public float freq() throws IOException {
            // We don't have the original child query hit info here...
            // But the freq of the children could be collector and returned here, but makes this Scorer more expensive.
            return 1;
        }

        @Override
        public int docID() {
            return currentDocId;
        }

        @Override
        public int nextDoc() throws IOException {
            while (true) {
                currentDocId = parentsIterator.nextDoc();
                if (currentDocId == DocIdSetIterator.NO_MORE_DOCS) {
                    return currentDocId;
                }

                HashedBytesArray uid = idTypeCache.idByDoc(currentDocId);
                currentScore = uidToScore.get(uid);
                if (Float.compare(currentScore, 0) > 0) {
                    return currentDocId;
                }
            }
        }

        @Override
        public int advance(int target) throws IOException {
            currentDocId = parentsIterator.advance(target);
            if (currentDocId == DocIdSetIterator.NO_MORE_DOCS) {
                return currentDocId;
            }

            HashedBytesArray uid = idTypeCache.idByDoc(currentDocId);
            currentScore = uidToScore.get(uid);
            if (Float.compare(currentScore, 0) > 0) {
                return currentDocId;
            } else {
                return nextDoc();
            }
        }
    }

    static class AvgParentScorer extends ParentScorer {

        final TObjectIntMap<HashedBytesArray> uidToCount;
        HashedBytesArray currentUid;

        AvgParentScorer(Weight weight, IdReaderTypeCache idTypeCache, TObjectFloatMap<HashedBytesArray> uidToScore, TObjectIntMap<HashedBytesArray> uidToCount, DocIdSetIterator parentsIterator) {
            super(weight, idTypeCache, uidToScore, parentsIterator);
            this.uidToCount = uidToCount;
        }

        @Override
        public int nextDoc() throws IOException {
            while (true) {
                currentDocId = parentsIterator.nextDoc();
                if (currentDocId == DocIdSetIterator.NO_MORE_DOCS) {
                    return currentDocId;
                }

                currentUid = idTypeCache.idByDoc(currentDocId);
                currentScore = uidToScore.get(currentUid);
                if (Float.compare(currentScore, 0) > 0) {
                    currentScore /= uidToCount.get(currentUid);
                    return currentDocId;
                }
            }
        }

        @Override
        public int advance(int target) throws IOException {
            currentDocId = parentsIterator.advance(target);
            if (currentDocId == DocIdSetIterator.NO_MORE_DOCS) {
                return currentDocId;
            }

            HashedBytesArray uid = idTypeCache.idByDoc(currentDocId);
            currentScore = uidToScore.get(uid);
            if (Float.compare(currentScore, 0) > 0) {
                currentScore /= uidToCount.get(currentUid);
                return currentDocId;
            } else {
                return nextDoc();
            }
        }
    }

    static class ChildUidCollector extends ParentIdCollector {

        final TObjectFloatHashMap<HashedBytesArray> uidToScore;
        final ScoreType scoreType;
        Scorer scorer;

        ChildUidCollector(ScoreType scoreType, SearchContext searchContext, String childType, TObjectFloatHashMap<HashedBytesArray> uidToScore) {
            super(childType, searchContext);
            this.uidToScore = uidToScore;
            this.scoreType = scoreType;
        }

        @Override
        public void setScorer(Scorer scorer) throws IOException {
            this.scorer = scorer;
        }

        @Override
        protected void collect(int doc, HashedBytesArray parentUid) throws IOException {
            float previousScore = uidToScore.get(parentUid);
            float currentScore = scorer.score();
            if (Float.compare(previousScore, 0) == 0) {
                uidToScore.put(parentUid, currentScore);
            } else {
                switch (scoreType) {
                    case SUM:
                        uidToScore.adjustValue(parentUid, currentScore);
                        break;
                    case MAX:
                        if (Float.compare(previousScore, currentScore) < 0) {
                            uidToScore.put(parentUid, currentScore);
                        }
                        break;
                    case AVG:
                        assert false : "AVG has it's own collector";
                        
                    default:
                        assert false : "Are we missing a sore type here? -- " + scoreType;
                        break;
                }
            }
        }
    }

    final static class AvgChildUidCollector extends ChildUidCollector {

        final TObjectIntHashMap<HashedBytesArray> uidToCount;

        AvgChildUidCollector(ScoreType scoreType, SearchContext searchContext, String childType, TObjectFloatHashMap<HashedBytesArray> uidToScore, TObjectIntHashMap<HashedBytesArray> uidToCount) {
            super(scoreType, searchContext, childType, uidToScore);
            this.uidToCount = uidToCount;
            assert scoreType == ScoreType.AVG;
        }

        @Override
        protected void collect(int doc, HashedBytesArray parentUid) throws IOException {
            float previousScore = uidToScore.get(parentUid);
            float currentScore = scorer.score();
            if (Float.compare(previousScore, 0) == 0) {
                uidToScore.put(parentUid, currentScore);
                uidToCount.put(parentUid, 1);
            } else {
                uidToScore.adjustValue(parentUid, currentScore);
                uidToCount.increment(parentUid);
            }
        }

    }

}
