/*
 * Licensed to ElasticSearch and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. ElasticSearch licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.indices.recovery;

import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.transport.TransportRequest;

import java.io.IOException;

/**
 *
 */
class RecoveryFileChunkRequest extends TransportRequest {

    private long recoveryId;
    private ShardId shardId;
    private String name;
    private long position;
    private long length;
    private String checksum;
    private BytesReference content;

    RecoveryFileChunkRequest() {
    }

    RecoveryFileChunkRequest(long recoveryId, ShardId shardId, String name, long position, long length, String checksum, BytesArray content) {
        this.recoveryId = recoveryId;
        this.shardId = shardId;
        this.name = name;
        this.position = position;
        this.length = length;
        this.checksum = checksum;
        this.content = content;
    }

    public long recoveryId() {
        return this.recoveryId;
    }

    public ShardId shardId() {
        return shardId;
    }

    public String name() {
        return name;
    }

    public long position() {
        return position;
    }

    @Nullable
    public String checksum() {
        return this.checksum;
    }

    public long length() {
        return length;
    }

    public BytesReference content() {
        return content;
    }

    public RecoveryFileChunkRequest readFileChunk(StreamInput in) throws IOException {
        RecoveryFileChunkRequest request = new RecoveryFileChunkRequest();
        request.readFrom(in);
        return request;
    }

    @Override
    public void readFrom(StreamInput in) throws IOException {
        super.readFrom(in);
        recoveryId = in.readLong();
        shardId = ShardId.readShardId(in);
        name = in.readString();
        position = in.readVLong();
        length = in.readVLong();
        checksum = in.readOptionalString();
        content = in.readBytesReference();
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        super.writeTo(out);
        out.writeLong(recoveryId);
        shardId.writeTo(out);
        out.writeString(name);
        out.writeVLong(position);
        out.writeVLong(length);
        out.writeOptionalString(checksum);
        out.writeBytesReference(content);
    }

    @Override
    public String toString() {
        return shardId + ": name='" + name + '\'' +
                ", position=" + position +
                ", length=" + length;
    }
}
