/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.security.authentication.ott;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jspecify.annotations.Nullable;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.jdbc.core.ArgumentPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcOperations;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SqlParameterValue;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.security.authentication.ott.DefaultOneTimeToken;
import org.springframework.security.authentication.ott.GenerateOneTimeTokenRequest;
import org.springframework.security.authentication.ott.OneTimeToken;
import org.springframework.security.authentication.ott.OneTimeTokenAuthenticationToken;
import org.springframework.security.authentication.ott.OneTimeTokenService;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

public final class JdbcOneTimeTokenService
implements OneTimeTokenService,
DisposableBean,
InitializingBean {
    private final Log logger = LogFactory.getLog(this.getClass());
    private final JdbcOperations jdbcOperations;
    private Function<OneTimeToken, List<SqlParameterValue>> oneTimeTokenParametersMapper = new OneTimeTokenParametersMapper();
    private RowMapper<OneTimeToken> oneTimeTokenRowMapper = new OneTimeTokenRowMapper();
    private Clock clock = Clock.systemUTC();
    private @Nullable ThreadPoolTaskScheduler taskScheduler;
    private static final String DEFAULT_CLEANUP_CRON = "@hourly";
    private static final String TABLE_NAME = "one_time_tokens";
    private static final String COLUMN_NAMES = "token_value, username, expires_at";
    private static final String SAVE_ONE_TIME_TOKEN_SQL = "INSERT INTO one_time_tokens (token_value, username, expires_at) VALUES (?, ?, ?)";
    private static final String FILTER = "token_value = ?";
    private static final String DELETE_ONE_TIME_TOKEN_SQL = "DELETE FROM one_time_tokens WHERE token_value = ?";
    private static final String SELECT_ONE_TIME_TOKEN_SQL = "SELECT token_value, username, expires_at FROM one_time_tokens WHERE token_value = ?";
    private static final String DELETE_ONE_TIME_TOKENS_BY_EXPIRY_TIME_QUERY = "DELETE FROM one_time_tokens WHERE expires_at < ?";

    public JdbcOneTimeTokenService(JdbcOperations jdbcOperations) {
        Assert.notNull((Object)jdbcOperations, (String)"jdbcOperations cannot be null");
        this.jdbcOperations = jdbcOperations;
        this.taskScheduler = this.createTaskScheduler(DEFAULT_CLEANUP_CRON);
    }

    public void setCleanupCron(String cleanupCron) {
        this.taskScheduler = this.createTaskScheduler(cleanupCron);
    }

    @Override
    public OneTimeToken generate(GenerateOneTimeTokenRequest request) {
        Assert.notNull((Object)request, (String)"generateOneTimeTokenRequest cannot be null");
        String token = UUID.randomUUID().toString();
        Instant expiresAt = this.clock.instant().plus(request.getExpiresIn());
        DefaultOneTimeToken oneTimeToken = new DefaultOneTimeToken(token, request.getUsername(), expiresAt);
        this.insertOneTimeToken(oneTimeToken);
        return oneTimeToken;
    }

    private void insertOneTimeToken(OneTimeToken oneTimeToken) {
        List<SqlParameterValue> parameters = this.oneTimeTokenParametersMapper.apply(oneTimeToken);
        ArgumentPreparedStatementSetter pss = new ArgumentPreparedStatementSetter(parameters.toArray());
        this.jdbcOperations.update(SAVE_ONE_TIME_TOKEN_SQL, (PreparedStatementSetter)pss);
    }

    @Override
    public @Nullable OneTimeToken consume(OneTimeTokenAuthenticationToken authenticationToken) {
        Assert.notNull((Object)authenticationToken, (String)"authenticationToken cannot be null");
        List<OneTimeToken> tokens = this.selectOneTimeToken(authenticationToken);
        if (CollectionUtils.isEmpty(tokens)) {
            return null;
        }
        OneTimeToken token = tokens.get(0);
        this.deleteOneTimeToken(token);
        if (this.isExpired(token)) {
            return null;
        }
        return token;
    }

    private boolean isExpired(OneTimeToken ott) {
        return this.clock.instant().isAfter(ott.getExpiresAt());
    }

    private List<OneTimeToken> selectOneTimeToken(OneTimeTokenAuthenticationToken authenticationToken) {
        List<SqlParameterValue> parameters = List.of(new SqlParameterValue(12, (Object)authenticationToken.getTokenValue()));
        ArgumentPreparedStatementSetter pss = new ArgumentPreparedStatementSetter(parameters.toArray());
        return this.jdbcOperations.query(SELECT_ONE_TIME_TOKEN_SQL, (PreparedStatementSetter)pss, this.oneTimeTokenRowMapper);
    }

    private void deleteOneTimeToken(OneTimeToken oneTimeToken) {
        List<SqlParameterValue> parameters = List.of(new SqlParameterValue(12, (Object)oneTimeToken.getTokenValue()));
        ArgumentPreparedStatementSetter pss = new ArgumentPreparedStatementSetter(parameters.toArray());
        this.jdbcOperations.update(DELETE_ONE_TIME_TOKEN_SQL, (PreparedStatementSetter)pss);
    }

    private @Nullable ThreadPoolTaskScheduler createTaskScheduler(String cleanupCron) {
        if (cleanupCron == null) {
            return null;
        }
        ThreadPoolTaskScheduler taskScheduler = new ThreadPoolTaskScheduler();
        taskScheduler.setThreadNamePrefix("spring-one-time-tokens-");
        taskScheduler.initialize();
        taskScheduler.schedule(this::cleanupExpiredTokens, (Trigger)new CronTrigger(cleanupCron));
        return taskScheduler;
    }

    public void cleanupExpiredTokens() {
        List<SqlParameterValue> parameters = List.of(new SqlParameterValue(93, (Object)Timestamp.from(Instant.now())));
        ArgumentPreparedStatementSetter pss = new ArgumentPreparedStatementSetter(parameters.toArray());
        int deletedCount = this.jdbcOperations.update(DELETE_ONE_TIME_TOKENS_BY_EXPIRY_TIME_QUERY, (PreparedStatementSetter)pss);
        if (this.logger.isDebugEnabled()) {
            this.logger.debug((Object)("Cleaned up " + deletedCount + " expired tokens"));
        }
    }

    public void afterPropertiesSet() throws Exception {
        if (this.taskScheduler != null) {
            this.taskScheduler.afterPropertiesSet();
        }
    }

    public void destroy() throws Exception {
        if (this.taskScheduler != null) {
            this.taskScheduler.shutdown();
        }
    }

    public void setClock(Clock clock) {
        Assert.notNull((Object)clock, (String)"clock cannot be null");
        this.clock = clock;
    }

    private static class OneTimeTokenParametersMapper
    implements Function<OneTimeToken, List<SqlParameterValue>> {
        private OneTimeTokenParametersMapper() {
        }

        @Override
        public List<SqlParameterValue> apply(OneTimeToken oneTimeToken) {
            ArrayList<SqlParameterValue> parameters = new ArrayList<SqlParameterValue>();
            parameters.add(new SqlParameterValue(12, (Object)oneTimeToken.getTokenValue()));
            parameters.add(new SqlParameterValue(12, (Object)oneTimeToken.getUsername()));
            parameters.add(new SqlParameterValue(93, (Object)Timestamp.from(oneTimeToken.getExpiresAt())));
            return parameters;
        }
    }

    private static class OneTimeTokenRowMapper
    implements RowMapper<OneTimeToken> {
        private OneTimeTokenRowMapper() {
        }

        public OneTimeToken mapRow(ResultSet rs, int rowNum) throws SQLException {
            String tokenValue = rs.getString("token_value");
            String userName = rs.getString("username");
            Instant expiresAt = rs.getTimestamp("expires_at").toInstant();
            return new DefaultOneTimeToken(tokenValue, userName, expiresAt);
        }
    }
}

