/*
 * =============================================================================
 * 
 *   Copyright (c) 2011-2016, The THYMELEAF team (http://www.thymeleaf.org)
 * 
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * 
 * =============================================================================
 */
package org.thymeleaf.model;

import java.io.IOException;
import java.io.Writer;

/**
 * <p>
 *   Common interface for all template events generated by the parsers (and processed by the
 *   implementations of {@link org.thymeleaf.engine.ITemplateHandler}.
 * </p>
 * <p>
 *   Sequences of these events are used to represent templates or fragments of them, by means of
 *   implementations of the {@link IModel} interface.
 * </p>
 * <p>
 *   Note that any implementations of this interface should be <strong>immutable</strong>.
 * </p>
 *
 * @author Daniel Fern&aacute;ndez
 * @since 3.0.0
 * 
 */
public interface ITemplateEvent {

    /**
     * <p>
     *   Checks whether this event contains location information (template name, line and column).
     * </p>
     * <p>
     *   Only events that are generated during the parsing of templates contain location info, locating them
     *   in their original template. All events generated during template processing and not originally present
     *   at the template do not contain this location data.
     * </p>
     *
     * @return whether the event contains location data or not.
     */
    public boolean hasLocation();

    /**
     * <p>
     *   Returns the name of the template from which parsing this event was originally created.
     * </p>
     *
     * @return the name of the template
     */
    public String getTemplateName();

    /**
     * <p>
     *   Returns the line at which this event can be found in the template specified by {@link #getTemplateName()}.
     * </p>
     *
     * @return the line number, starting in 1.
     */
    public int getLine();

    /**
     * <p>
     *   Returns the column at which this event can be found in the template specified by {@link #getTemplateName()}.
     * </p>
     *
     * @return the column number, starting in 1.
     */
    public int getCol();


    /**
     * <p>
     *   Accept a visitor, implementation of {@link IModelVisitor}.
     * </p>
     *
     * @param visitor the visitor.
     */
    public void accept(final IModelVisitor visitor);


    /**
     * <p>
     *   Writes this event to the specified {@link Writer}.
     * </p>
     * <p>
     *   Template output performed at {@link org.thymeleaf.engine.OutputTemplateHandler} is done by calling
     *   these methods at each of the events resulting from template processing.
     * </p>
     *
     * @param writer the writer this event should be written to.
     * @throws IOException if an input/output exception occurs.
     */
    public void write(final Writer writer) throws IOException;

}
