/*
 * Decompiled with CFR 0.152.
 */
package tech.powerjob.server.common.timewheel;

import com.google.common.collect.Queues;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.powerjob.common.utils.CommonUtils;
import tech.powerjob.server.common.RejectedExecutionHandlerFactory;
import tech.powerjob.server.common.timewheel.Timer;
import tech.powerjob.server.common.timewheel.TimerFuture;
import tech.powerjob.server.common.timewheel.TimerTask;

public class HashedWheelTimer
implements Timer {
    private static final Logger log = LoggerFactory.getLogger(HashedWheelTimer.class);
    private final long tickDuration;
    private final HashedWheelBucket[] wheel;
    private final int mask;
    private final Indicator indicator;
    private final long startTime;
    private final Queue<HashedWheelTimerFuture> waitingTasks = Queues.newLinkedBlockingQueue();
    private final Queue<HashedWheelTimerFuture> canceledTasks = Queues.newLinkedBlockingQueue();
    private final ExecutorService taskProcessPool;

    public HashedWheelTimer(long tickDuration, int ticksPerWheel) {
        this(tickDuration, ticksPerWheel, 0);
    }

    public HashedWheelTimer(long tickDuration, int ticksPerWheel, int processThreadNum) {
        this.tickDuration = tickDuration;
        int ticksNum = CommonUtils.formatSize((int)ticksPerWheel);
        this.wheel = new HashedWheelBucket[ticksNum];
        for (int i = 0; i < ticksNum; ++i) {
            this.wheel[i] = new HashedWheelBucket();
        }
        this.mask = this.wheel.length - 1;
        if (processThreadNum <= 0) {
            this.taskProcessPool = null;
        } else {
            ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("HashedWheelTimer-Executor-%d").build();
            LinkedBlockingQueue queue = Queues.newLinkedBlockingQueue((int)8192);
            int core = Math.max(Runtime.getRuntime().availableProcessors(), processThreadNum);
            this.taskProcessPool = new ThreadPoolExecutor(core, 2 * core, 60L, TimeUnit.SECONDS, queue, threadFactory, RejectedExecutionHandlerFactory.newCallerRun("PowerJobTimeWheelPool"));
        }
        this.startTime = System.currentTimeMillis();
        this.indicator = new Indicator();
        new Thread((Runnable)this.indicator, "HashedWheelTimer-Indicator").start();
    }

    @Override
    public TimerFuture schedule(TimerTask task, long delay, TimeUnit unit) {
        long targetTime = System.currentTimeMillis() + unit.toMillis(delay);
        HashedWheelTimerFuture timerFuture = new HashedWheelTimerFuture(task, targetTime);
        if (delay <= 0L) {
            this.runTask(timerFuture);
            return timerFuture;
        }
        this.waitingTasks.add(timerFuture);
        return timerFuture;
    }

    @Override
    public Set<TimerTask> stop() {
        this.indicator.stop.set(true);
        this.taskProcessPool.shutdown();
        while (!this.taskProcessPool.isTerminated()) {
            try {
                Thread.sleep(100L);
            }
            catch (Exception exception) {}
        }
        return this.indicator.getUnprocessedTasks();
    }

    private void runTask(HashedWheelTimerFuture timerFuture) {
        timerFuture.status = 1;
        if (this.taskProcessPool == null) {
            timerFuture.timerTask.run();
        } else {
            this.taskProcessPool.submit(timerFuture.timerTask);
        }
    }

    private class Indicator
    implements Runnable {
        private long tick = 0L;
        private final AtomicBoolean stop = new AtomicBoolean(false);
        private final CountDownLatch latch = new CountDownLatch(1);

        private Indicator() {
        }

        @Override
        public void run() {
            while (!this.stop.get()) {
                this.pushTaskToBucket();
                this.processCanceledTasks();
                this.tickTack();
                int currentIndex = (int)(this.tick & (long)HashedWheelTimer.this.mask);
                HashedWheelBucket bucket = HashedWheelTimer.this.wheel[currentIndex];
                bucket.expireTimerTasks(this.tick);
                ++this.tick;
            }
            this.latch.countDown();
        }

        private void tickTack() {
            long nextTime = HashedWheelTimer.this.startTime + (this.tick + 1L) * HashedWheelTimer.this.tickDuration;
            long sleepTime = nextTime - System.currentTimeMillis();
            if (sleepTime > 0L) {
                try {
                    Thread.sleep(sleepTime);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }

        private void processCanceledTasks() {
            HashedWheelTimerFuture canceledTask;
            while ((canceledTask = (HashedWheelTimerFuture)HashedWheelTimer.this.canceledTasks.poll()) != null) {
                if (canceledTask.bucket == null) continue;
                canceledTask.bucket.remove(canceledTask);
            }
            return;
        }

        private void pushTaskToBucket() {
            HashedWheelTimerFuture timerTask;
            while ((timerTask = (HashedWheelTimerFuture)HashedWheelTimer.this.waitingTasks.poll()) != null) {
                long offset = timerTask.targetTime - HashedWheelTimer.this.startTime;
                timerTask.totalTicks = offset / HashedWheelTimer.this.tickDuration;
                int index = (int)(timerTask.totalTicks & (long)HashedWheelTimer.this.mask);
                HashedWheelBucket bucket = HashedWheelTimer.this.wheel[index];
                timerTask.bucket = bucket;
                if (timerTask.status != 0) continue;
                bucket.add(timerTask);
            }
            return;
        }

        public Set<TimerTask> getUnprocessedTasks() {
            try {
                this.latch.await();
            }
            catch (Exception exception) {
                // empty catch block
            }
            HashSet tasks = Sets.newHashSet();
            Consumer<HashedWheelTimerFuture> consumer = timerFuture -> {
                if (((HashedWheelTimerFuture)timerFuture).status == 0) {
                    tasks.add(((HashedWheelTimerFuture)timerFuture).timerTask);
                }
            };
            HashedWheelTimer.this.waitingTasks.forEach(consumer);
            for (HashedWheelBucket bucket : HashedWheelTimer.this.wheel) {
                bucket.forEach(consumer);
            }
            return tasks;
        }
    }

    private final class HashedWheelBucket
    extends LinkedList<HashedWheelTimerFuture> {
        private HashedWheelBucket() {
        }

        public void expireTimerTasks(long currentTick) {
            this.removeIf(timerFuture -> {
                if (((HashedWheelTimerFuture)timerFuture).status == 3) {
                    return true;
                }
                if (((HashedWheelTimerFuture)timerFuture).status != 0) {
                    log.warn("[HashedWheelTimer] impossible, please fix the bug");
                    return true;
                }
                if (((HashedWheelTimerFuture)timerFuture).totalTicks <= currentTick) {
                    if (((HashedWheelTimerFuture)timerFuture).totalTicks < currentTick) {
                        log.warn("[HashedWheelTimer] timerFuture.totalTicks < currentTick, please fix the bug");
                    }
                    try {
                        HashedWheelTimer.this.runTask(timerFuture);
                    }
                    catch (Exception exception) {
                    }
                    finally {
                        ((HashedWheelTimerFuture)timerFuture).status = 2;
                    }
                    return true;
                }
                return false;
            });
        }
    }

    private final class HashedWheelTimerFuture
    implements TimerFuture {
        private final long targetTime;
        private final TimerTask timerTask;
        private HashedWheelBucket bucket;
        private long totalTicks;
        private int status;
        private static final int WAITING = 0;
        private static final int RUNNING = 1;
        private static final int FINISHED = 2;
        private static final int CANCELED = 3;

        public HashedWheelTimerFuture(TimerTask timerTask, long targetTime) {
            this.targetTime = targetTime;
            this.timerTask = timerTask;
            this.status = 0;
        }

        @Override
        public TimerTask getTask() {
            return this.timerTask;
        }

        @Override
        public boolean cancel() {
            if (this.status == 0) {
                this.status = 3;
                HashedWheelTimer.this.canceledTasks.add(this);
                return true;
            }
            return false;
        }

        @Override
        public boolean isCancelled() {
            return this.status == 3;
        }

        @Override
        public boolean isDone() {
            return this.status == 2;
        }
    }
}

