package HslCommunication.CNC.Fanuc;

import HslCommunication.Authorization;
import HslCommunication.BasicFramework.SoftBasic;
import HslCommunication.Core.IMessage.INetMessage;
import HslCommunication.Core.Net.NetworkBase.NetworkDoubleBase;
import HslCommunication.Core.Transfer.ByteTransformHelper;
import HslCommunication.Core.Transfer.ReverseBytesTransform;
import HslCommunication.Core.Types.OperateResult;
import HslCommunication.Core.Types.OperateResultExOne;
import HslCommunication.Core.Types.OperateResultExTwo;
import HslCommunication.StringResources;
import HslCommunication.Utilities;

import java.io.*;
import java.math.BigDecimal;
import java.net.Socket;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Date;

public class FanucSeries0i extends NetworkDoubleBase {
    /**
     * 根据IP及端口来实例化一个对象内容，端口一般默认为 8193
     *
     * @param ipAddress Ip地址信息
     * @param port      端口号
     */
    public FanucSeries0i(String ipAddress, int port) {
        this.setIpAddress(ipAddress);
        this.setPort(port);
        this.setByteTransform(new ReverseBytesTransform());
        this.encoding = Charset.forName("GB2312");
        this.setReceiveTimeOut(30_000);
    }

    protected INetMessage GetNewNetMessage() {
        return new CNCFanucSeriesMessage();
    }

    /**
     * 获取当前的文本的字符编码信息<br />
     *
     * @return 文本编码信息
     */
    public Charset getTextEncoding() {
        return this.encoding;
    }

    /**
     * 设置当前的文本的字符编码信息，如果你不清楚，可以调用  方法来自动匹配。
     *
     * @param value 设置了新的值
     */
    public void setTextEncoding(Charset value) {
        this.encoding = value;
    }

    protected OperateResult InitializationOnConnect(Socket socket) {
        OperateResultExOne<byte[]> read1 = ReadFromCoreServer(socket, SoftBasic.HexStringToBytes("a0 a0 a0 a0 00 01 01 01 00 02 00 02"), true, true);
        if (!read1.IsSuccess) return read1;

        OperateResultExOne<byte[]> read2 = ReadFromCoreServer(socket, SoftBasic.HexStringToBytes("a0 a0 a0 a0 00 01 21 01 00 1e 00 01 00 1c 00 01 00 01 00 18 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00"), true, true);
        if (!read2.IsSuccess) return read2;

        this.fanucSysInfo = new FanucSysInfo( read2.Content );
        return OperateResult.CreateSuccessResult();
    }

    protected OperateResult ExtraOnDisconnect(Socket socket) {
        return ReadFromCoreServer(socket, SoftBasic.HexStringToBytes("a0 a0 a0 a0 00 01 02 01 00 00"), true, true);
    }

    private double GetFanucDouble(byte[] content, int index) {
        return GetFanucDouble(content, index, 1)[0];
    }

    private double[] GetFanucDouble(byte[] content, int index, int length) {
        double[] buffer = new double[length];
        for (int i = 0; i < length; i++) {
            int data = getByteTransform().TransInt32(content, index + 8 * i);
            int decs = getByteTransform().TransInt16(content, index + 8 * i + 6);

            if (data == 0)
                buffer[i] = 0;
            else
                buffer[i] = new BigDecimal(data * Math.pow(0.1d, decs)).setScale(decs, BigDecimal.ROUND_HALF_UP).doubleValue();
        }
        return buffer;
    }

    private byte[] CreateFromFanucDouble(double value) {
        byte[] buffer = new byte[8];
        int interge = (int) (value * 1000);

        System.arraycopy(getByteTransform().TransByte(interge), 0, buffer, 0, 4);
        buffer[5] = 0x0A;
        buffer[7] = 0x03;
        return buffer;
    }

    private void ChangeTextEncoding(short code) {
        switch (code) {
            case 0x00:
                this.encoding = Charset.forName("GB2312");
                break;
            case 0x01:
            case 0x04:
                this.encoding = Charset.forName("shift_jis");
                break;
            case 0x06:
                this.encoding = Charset.forName("ks_c_5601-1987");
                break;
            case 0x0F:
                this.encoding = Charset.forName("GB2312");
                break;
            case 0x10:
                this.encoding = Charset.forName("windows-1251");
                break;
            case 0x11:
                this.encoding = Charset.forName("windows-1254");
                break;
        }
    }

    /**
     * 主轴转速及进给倍率<br />
     * Spindle speed and feedrate override
     *
     * @return 主轴转速及进给倍率
     */
    public OperateResultExTwo<Double, Double> ReadSpindleSpeedAndFeedRate() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0xA4, 3, 0, 0, 0, 0),
                BuildReadSingle((short) 0x8A, 1, 0, 0, 0, 0),
                BuildReadSingle((short) 0x88, 3, 0, 0, 0, 0),
                BuildReadSingle((short) 0x88, 4, 0, 0, 0, 0),
                BuildReadSingle((short) 0x24, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0x25, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0xA4, 3, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExTwo.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        return OperateResultExTwo.CreateSuccessResult(GetFanucDouble(result.get(5), 14), GetFanucDouble(result.get(4), 14));
    }

    /**
     * 获取fanuc机床设备的基本信息，型号，轴数量等等
     * @return 机床信息
     */
    public OperateResultExOne<FanucSysInfo> ReadSysInfo( )
    {
        return this.fanucSysInfo == null ? new OperateResultExOne<FanucSysInfo>( "Must connect device first!" ) : OperateResultExOne.CreateSuccessResult( this.fanucSysInfo );
    }

    private OperateResult CheckSingleResultLeagle( byte[] result )
    {
        int status = result[6] * 256 + result[7];
        if (status != 0x00) return new OperateResultExOne<>( status, StringResources.Language.UnknownError() );

        return OperateResult.CreateSuccessResult( );
    }

    /**
     * 读取指定路径下的所有的子路径和文件的信息，路径信息，例如 "//CNC_MEM/USER/"
     * @param path 路径信息，例如 "//CNC_MEM/USER/"
     * @return 文件及路径信息
     */
    public OperateResultExOne<FileDirInfo[]> ReadAllDirectoryAndFile( String path )
    {
        if (!path.endsWith( "/" )) path += "/";

        byte[] buffer = new byte[256];
        byte[] pathBuffer = path.getBytes(StandardCharsets.US_ASCII);
        System.arraycopy(pathBuffer, 0, buffer, 0, pathBuffer.length);

        // 先获取文件数量信息
        OperateResultExOne<Integer> readCount = ReadAllDirectoryAndFileCount( path );
        if (!readCount.IsSuccess) return OperateResultExOne.CreateFailedResult( readCount );
        if (readCount.Content == 0) return OperateResultExOne.CreateSuccessResult( new FileDirInfo[0] );

        // 分割成20的长度读取
        int[] splits = SoftBasic.SplitIntegerToArray( readCount.Content, 0x14 );
        ArrayList<FileDirInfo> list = new ArrayList<>( );

        int already = 0;
        for ( int j = 0; j < splits.length; j++)
        {
            // a:req_num, b:num_prog, c:type, d:size_kind
            OperateResultExOne<byte[]> read = ReadFromCoreServer( BuildReadArray( BuildWriteSingle( (short)0x01, (short) 0xB3, already, splits[j], 1, 1, buffer ) ) );
            if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult( read );

            if (read.Content.length == 0x12 || this.getByteTransform().TransInt16( read.Content, 10 ) == 0)
            {
                read = ReadFromCoreServer( BuildReadArray( BuildWriteSingle( (short) 0x01, (short) 0xB3, 0, 0x14, 1, 1, buffer ) ) );
                if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult( read );
            }

            byte[] result = ExtraContentArray( SoftBasic.BytesArrayRemoveBegin(read.Content, 10 ) ).get(0);

            OperateResult checkResult = CheckSingleResultLeagle( result );
            if (!checkResult.IsSuccess) return OperateResultExOne.CreateFailedResult( checkResult );

            int count = (result.length - 14) / 128;
            for (int i = 0; i < count; i++)
            {
                list.add( new FileDirInfo( this.getByteTransform(), result, 14 + 128 * i ) );
            }
            already += splits[j];
        }
        return OperateResultExOne.CreateSuccessResult( Utilities.ArrayListToArray( FileDirInfo.class, list ) );
    }

    /**
     * 获取指定的路径里所有的文件夹数量和文件数量之和，路径示例：例如 "//CNC_MEM/USER/"， "//CNC_MEM/USER/PATH1/"
     * @param path 路径信息，例如 "//CNC_MEM/USER/"
     * @return 文件夹数量和文件数量之和
     */
    public OperateResultExOne<Integer> ReadAllDirectoryAndFileCount( String path )
    {
        if (!path.endsWith( "/" )) path += "/";

        byte[] buffer = new byte[256];
        byte[] pathBuffer = path.getBytes(StandardCharsets.US_ASCII);
        System.arraycopy(pathBuffer, 0, buffer, 0, pathBuffer.length);

        OperateResultExOne<byte[]> read = ReadFromCoreServer( BuildReadArray( BuildWriteSingle( (short) 0x01, (short) 0xB4, 0, 0, 0, 0, buffer ) ) );
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult( read );

        byte[] result = ExtraContentArray( SoftBasic.BytesArrayRemoveBegin(read.Content, 10 ) ).get(0);

        OperateResult checkResult = CheckSingleResultLeagle( result );
        if (!checkResult.IsSuccess) return OperateResultExOne.CreateFailedResult( checkResult );

        return OperateResultExOne.CreateSuccessResult( getByteTransform().TransInt32( result, 14 ) + getByteTransform().TransInt32( result, 18 ) );
    }

    /**
     * 读取程序名及程序号<br />
     * Read program name and program number
     * @return 程序名及程序号
     */
    public OperateResultExTwo<String, Integer> ReadSystemProgramCurrent() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0xCF, 0, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExTwo.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int number = getByteTransform().TransInt32(result.get(0), 14);
        String name = new String(SoftBasic.BytesArraySelectMiddle(result.get(0), 18, 36), this.encoding).trim();
        return OperateResultExTwo.CreateSuccessResult(name, number);
    }

    /**
     * 读取机床的语言设定信息，具体值的含义参照API文档说明<br />
     * Read the language setting information of the machine tool, refer to the API documentation for the meaning of the specific values
     * @return 此处举几个常用值 0: 英语 1: 日语 2: 德语 3: 法语 4: 中文繁体 6: 韩语 15: 中文简体 16: 俄语 17: 土耳其语
     */
    public OperateResultExOne<Short> ReadLanguage() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x8D, 0x0CD1, 0x0CD1, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);
        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));

        short code = getByteTransform().TransInt16(result.get(0), 24);
        ChangeTextEncoding(code);
        return OperateResultExOne.CreateSuccessResult(code);
    }

    /**
     * 读取宏变量，可以用来读取刀具号<br />
     * Read macro variable, can be used to read tool number
     * @param number 刀具号
     * @return 读宏变量信息
     */
    public OperateResultExOne<Double> ReadSystemMacroValue(int number) {
        OperateResultExOne<double[]> read = ReadSystemMacroValue(number, 1);
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        return OperateResultExOne.CreateSuccessResult(read.Content[0]);
    }

    /**
     * 读取宏变量，可以用来读取刀具号<br />
     * Read macro variable, can be used to read tool number
     * @param number 宏变量地址
     * @param length 读取的长度信息
     * @return 是否成功
     */
    public OperateResultExOne<double[]> ReadSystemMacroValue(int number, int length) {
        // 拆分5个5个读
        int[] lenArray = SoftBasic.SplitIntegerToArray(length, 5);
        int index = number;
        ArrayList<Byte> result = new ArrayList<>();

        for (int i = 0; i < lenArray.length; i++) {
            OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                    BuildReadSingle((short) 0x15, index, index + lenArray[i] - 1, 0, 0, 0)));
            if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

            Utilities.ArrayListAddArray(result, SoftBasic.BytesArrayRemoveBegin(ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10)).get(0), 14));
            index += lenArray[i];
        }

        try {
            return OperateResultExOne.CreateSuccessResult(GetFanucDouble(Utilities.getBytes(result), 0, length));
        } catch (Exception ex) {
            return new OperateResultExOne<double[]>(ex.getMessage() + " Source:" + SoftBasic.ByteToHexString(Utilities.getBytes(result), ' '));
        }
    }

    /**
     * 写宏变量，需要指定地址及写入的数据<br />
     * Write macro variable, need to specify the address and write data
     * @param number 地址
     * @param values 数据值
     * @return 是否成功
     */
    public OperateResult WriteSystemMacroValue(int number, double[] values) {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildWriteSingle((short) 0x16, number, number + values.length - 1, 0, 0, values)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        if (getByteTransform().TransUInt16(result.get(0), 6) == 0) {
            return OperateResult.CreateSuccessResult();
        } else {
            return new OperateResult(getByteTransform().TransUInt16(result.get(0), 6), "Unknown Error");
        }
    }

    /**
     * 根据刀具号写入长度形状补偿，刀具号为1-24<br />
     * Write length shape compensation according to the tool number, the tool number is 1-24
     * @param cutter >刀具号，范围为1-24
     * @param offset 补偿值
     * @return 是否写入成功
     */
    public OperateResult WriteCutterLengthShapeOffset(int cutter, double offset) {
        return WriteSystemMacroValue(11000 + cutter, new double[]{offset});
    }

    /**
     * 根据刀具号写入长度磨损补偿，刀具号为1-24<br />
     * Write length wear compensation according to the tool number, the tool number is 1-24
     * @param cutter 刀具号，范围为1-24
     * @param offset 补偿值
     * @return 是否写入成功
     */
    public OperateResult WriteCutterLengthWearOffset(int cutter, double offset) {
        return WriteSystemMacroValue(10000 + cutter, new double[]{offset});
    }

    /**
     * 根据刀具号写入半径形状补偿，刀具号为1-24
     * Write radius shape compensation according to the tool number, the tool number is 1-24
     * @param cutter 刀具号，范围为1-24
     * @param offset 补偿值
     * @return 是否写入成功
     */
    public OperateResult WriteCutterRadiusShapeOffset(int cutter, double offset) {
        return WriteSystemMacroValue(13000 + cutter, new double[]{offset});
    }

    /**
     * 根据刀具号写入半径磨损补偿，刀具号为1-24<br />
     * Write radius wear compensation according to the tool number, the tool number is 1-24
     * @param cutter 刀具号，范围为1-24
     * @param offset 补偿值
     * @return 是否写入成功
     */
    public OperateResult WriteCutterRadiusWearOffset(int cutter, double offset) {
        return WriteSystemMacroValue(12000 + cutter, new double[]{offset});
    }

    /**
     * 读取伺服负载<br />
     * Read servo load
     * @return 轴负载
     */
    public OperateResultExOne<double[]> ReadFanucAxisLoad() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0xA4, 2, 0, 0, 0, 0),
                BuildReadSingle((short) 0x89, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0x56, 1, 0, 0, 0, 0),
                BuildReadSingle((short) 0xA4, 2, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int length = getByteTransform().TransUInt16(result.get(0), 14);

        return OperateResultExOne.CreateSuccessResult(GetFanucDouble(result.get(2), 14, length));
    }

    /**
     * 读取机床的坐标，包括机械坐标，绝对坐标，相对坐标<br />
     * Read the coordinates of the machine tool, including mechanical coordinates, absolute coordinates, and relative coordinates
     * @return 数控机床的坐标信息，包括机械坐标，绝对坐标，相对坐标
     */
    public OperateResultExOne<SysAllCoors> ReadSysAllCoors() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0xA4, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0x89, -1, 0, 0, 0, 0),
                BuildReadSingle((short) 0x88, 1, 0, 0, 0, 0),
                BuildReadSingle((short) 0x88, 2, 0, 0, 0, 0),
                BuildReadSingle((short) 0xA3, 0, -1, 0, 0, 0),
                BuildReadSingle((short) 0x26, 0, -1, 0, 0, 0),
                BuildReadSingle((short) 0x26, 1, -1, 0, 0, 0),
                BuildReadSingle((short) 0x26, 2, -1, 0, 0, 0),
                BuildReadSingle((short) 0x26, 3, -1, 0, 0, 0),
                BuildReadSingle((short) 0xA4, 0, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int length = getByteTransform().TransUInt16(result.get(0), 14);

        SysAllCoors allCoors = new SysAllCoors();

        allCoors.Absolute = GetFanucDouble(result.get(5), 14, length);
        allCoors.Machine = GetFanucDouble(result.get(6), 14, length);
        allCoors.Relative = GetFanucDouble(result.get(7), 14, length);

        return OperateResultExOne.CreateSuccessResult(allCoors);
    }

    /**
     * 读取报警信息<br />
     * Read alarm information
     * @return 机床的当前的所有的报警信息
     */
    public OperateResultExOne<SysAlarm[]> ReadSystemAlarm() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x23, -1, 10, 2, 64, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        if (getByteTransform().TransUInt16(result.get(0), 12) > 0) {
            int length = getByteTransform().TransUInt16(result.get(0), 12) / 80;
            SysAlarm[] alarms = new SysAlarm[length];
            for (int i = 0; i < alarms.length; i++) {
                alarms[i] = new SysAlarm();
                alarms[i].AlarmId = getByteTransform().TransInt32(result.get(0), 14 + 80 * i);
                alarms[i].Type = getByteTransform().TransInt16(result.get(0), 20 + 80 * i);
                alarms[i].Axis = getByteTransform().TransInt16(result.get(0), 24 + 80 * i);

                int msgLength = getByteTransform().TransUInt16(result.get(0), 28 + 80 * i);
                alarms[i].Message = new String(result.get(0), 30 + 80 * i, msgLength, this.encoding);
            }
            return OperateResultExOne.CreateSuccessResult(alarms);
        } else
            return OperateResultExOne.CreateSuccessResult(new SysAlarm[0]);
    }

    /**
     * 读取fanuc机床的时间，0是开机时间，1是运行时间，2是切割时间，3是循环时间，4是空闲时间，返回秒为单位的信息<br />
     * Read the time of the fanuc machine tool, 0 is the boot time, 1 is the running time, 2 is the cutting time,
     * 3 is the cycle time, 4 is the idle time, and returns the information in seconds.
     * @param timeType 读取的时间类型
     * @return 秒为单位的结果
     */
    public OperateResultExOne<Long> ReadTimeData(int timeType) {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x0120, timeType, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int millisecond = getByteTransform().TransInt32(result.get(0), 18);
        long munite     = getByteTransform().TransInt32(result.get(0), 14);

        if (millisecond < 0 || millisecond > 60000 || munite < 0) {
            millisecond = Utilities.getInt(result.get(0), 18);
            munite      = Utilities.getInt(result.get(0), 14);
        }

        long seconds = millisecond / 1000;

        return OperateResultExOne.CreateSuccessResult(munite * 60 + seconds);
    }

    /**
     * 读取报警状态信息<br />
     * Read alarm status information
     * @return 报警状态数据
     */
    public OperateResultExOne<Integer> ReadAlarmStatus() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x1A, 0, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        return OperateResultExOne.CreateSuccessResult((int) getByteTransform().TransUInt16(result.get(0), 16));
    }

    /**
     * 读取系统的基本信息状态，工作模式，运行状态，是否急停等等操作<br />
     * Read the basic information status of the system, working mode, running status, emergency stop, etc.
     * @return 结果信息数据
     */
    public OperateResultExOne<SysStatusInfo> ReadSysStatusInfo() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x19, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0xE1, 0, 0, 0, 0, 0),
                BuildReadSingle((short) 0x98, 0, 0, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        SysStatusInfo statusInfo = new SysStatusInfo();
        statusInfo.Dummy = getByteTransform().TransInt16(result.get(1), 14);
        statusInfo.TMMode = result.get(2).length >= 16 ? getByteTransform().TransInt16(result.get(2), 14) : (short) 0;
        statusInfo.WorkMode = CNCWorkMode.values()[getByteTransform().TransInt16(result.get(0), 14)];
        statusInfo.RunStatus = CNCRunStatus.values()[getByteTransform().TransInt16(result.get(0), 16)];
        statusInfo.Motion = getByteTransform().TransInt16(result.get(0), 18);
        statusInfo.MSTB = getByteTransform().TransInt16(result.get(0), 20);
        statusInfo.Emergency = getByteTransform().TransInt16(result.get(0), 22);
        statusInfo.Alarm = getByteTransform().TransInt16(result.get(0), 24);
        statusInfo.Edit = getByteTransform().TransInt16(result.get(0), 26);

        return OperateResultExOne.CreateSuccessResult(statusInfo);
    }

    /**
     * 读取设备的程序列表<br />
     * Read the program list of the device
     * @return 读取结果信息
     */
    public OperateResultExOne<int[]> ReadProgramList() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x06, 0x01, 0x13, 0, 0, 0)
        ));
        OperateResultExOne<byte[]> check = ReadFromCoreServer(BuildReadArray(
                BuildReadSingle((short) 0x06, 0x1A0B, 0x13, 0, 0, 0)
        ));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);
        if (!check.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int length = (result.get(0).length - 14) / 72;
        int[] programs = new int[length];
        for (int i = 0; i < length; i++) {
            programs[i] = getByteTransform().TransInt32(result.get(0), 14 + 72 * i);
        }
        return OperateResultExOne.CreateSuccessResult(programs);
    }

    /**
     * 读取当前的刀具补偿信息，默认刀具数量应该是24<br />
     * Read current tool compensation information
     * @param cutterNumber 刀具数量
     * @return 结果内容
     */
    public OperateResultExOne<CutterInfo[]> ReadCutterInfos(int cutterNumber) {
        OperateResultExOne<byte[]> read1 = ReadFromCoreServer(BuildReadArray(BuildReadSingle((short) 0x08, 1, cutterNumber, 0, 0, 0)));
        if (!read1.IsSuccess) return OperateResultExOne.CreateFailedResult(read1);

        OperateResultExOne<byte[]> read2 = ReadFromCoreServer(BuildReadArray(BuildReadSingle((short) 0x08, 1, cutterNumber, 1, 0, 0)));
        if (!read2.IsSuccess) return OperateResultExOne.CreateFailedResult(read2);

        OperateResultExOne<byte[]> read3 = ReadFromCoreServer(BuildReadArray(BuildReadSingle((short) 0x08, 1, cutterNumber, 2, 0, 0)));
        if (!read3.IsSuccess) return OperateResultExOne.CreateFailedResult(read3);

        OperateResultExOne<byte[]> read4 = ReadFromCoreServer(BuildReadArray(BuildReadSingle((short) 0x08, 1, cutterNumber, 3, 0, 0)));
        if (!read4.IsSuccess) return OperateResultExOne.CreateFailedResult(read4);

        return ExtraCutterInfos(read1.Content, read2.Content, read3.Content, read4.Content, cutterNumber);
    }

    /**
     * 读取当前的正在使用的刀具号<br />
     * Read the tool number currently in use
     * @return 刀具号信息
     */
    public OperateResultExOne<Integer> ReadCutterNumber() {
        OperateResultExOne<double[]> read = ReadSystemMacroValue(4120, 1);
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        return OperateResultExOne.CreateSuccessResult((new Double(read.Content[0]).intValue()));
    }

    /**
     * 读取R数据，需要传入起始地址和结束地址，返回byte[]数据信息<br />
     * To read R data, you need to pass in the start address and end address, and return byte[] data information
     * @param start 起始地址
     * @param end 结束地址
     * @return 读取结果
     */
    public OperateResultExOne<byte[]> ReadRData(int start, int end) {
        OperateResultExOne<byte[]> read1 = ReadFromCoreServer(
                BuildReadArray(BuildReadMulti((short) 0x02, (short) 0x8001, start, end, 0x05, 0, 0)));
        if (!read1.IsSuccess) return read1;

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read1.Content, 10));
        int length = this.getByteTransform().TransUInt16(result.get(0), 12);
        return OperateResultExOne.CreateSuccessResult(SoftBasic.BytesArraySelectMiddle(result.get(0), 14, length));
    }

    /**
     * 读取工件尺寸<br />
     * Read workpiece size
     * @return 结果数据信息
     */
    public OperateResultExOne<double[]> ReadDeviceWorkPiecesSize() {
        return ReadSystemMacroValue(601, 20);
    }

    /**
     * 读取当前的程序内容，只能读取程序的片段，返回程序内容。<br />
     *  Read the current program content, only read the program fragments, and return the program content.
     * @return 程序内容
     */
    public OperateResultExOne<String> ReadCurrentProgram() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(
                BuildReadArray(BuildReadSingle((short) 0x20, 0x0594, 0x00, 0x00, 0x00, 0x00)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        byte[] result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10)).get(0);
        return OperateResultExOne.CreateSuccessResult(new String(result, 18, result.length - 18, StandardCharsets.US_ASCII));
    }

    /**
     * 设置指定的程序号为当前的主程序，如果程序号不存在，返回错误信息<br />
     * Set the specified program number as the current main program, if the program number does not exist, an error message will be returned
     * @param programNum 程序号信息
     * @return 是否设置成功
     */
    public OperateResult SetCurrentProgram(short programNum) {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(
                BuildReadArray(BuildReadSingle((short) 0x03, programNum, 0, 0, 0, 0)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        byte[] result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10)).get(0);
        short err = this.getByteTransform().TransInt16(result, 6);

        if (err == 0) return OperateResult.CreateSuccessResult();
        else return new OperateResult(err, StringResources.Language.UnknownError());
    }

    /**
     * 启动加工程序<br />
     * Start the processing program
     * @return 是否启动成功
     */
    public OperateResult StartProcessing() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(
                BuildReadArray(BuildReadSingle((short) 0x01, 0, 0, 0, 0, 0)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        byte[] result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10)).get(0);
        short err = this.getByteTransform().TransInt16(result, 6);

        if (err == 0) return OperateResult.CreateSuccessResult();
        else return new OperateResult(err, StringResources.Language.UnknownError());
    }

    /**
     * <b>[商业授权]</b> 将指定文件的NC加工程序，下载到数控机床里，返回是否下载成功<br />
     * <b>[Authorization]</b> Download the NC machining program of the specified file to the CNC machine tool, and return whether the download is successful<br />
     * <br />
     * 程序文件的内容必须%开始，%结束，下面是一个非常简单的例子：<br />
     * %<br />
     * O0006<br />
     * G90G10L2P1<br />
     * M30<br />
     * %
     * @param file 程序文件的路径
     * @return 是否下载成功
     * @throws IOException 文件异常
     */
    public OperateResult WriteProgramFile(String file) throws IOException {
        String content = new String(Files.readAllBytes(Paths.get(file)), StandardCharsets.US_ASCII);
        return WriteProgramContent(content, 512);
    }

    /**
     * <b>[商业授权]</b> 将指定文件的NC加工程序，下载到数控机床里，返回是否下载成功<br />
     * <b>[Authorization]</b> Download the NC machining program of the specified file to the CNC machine tool, and return whether the download is successful<br />
     * <br />
     * 程序文件的内容必须%开始，%结束，下面是一个非常简单的例子：<br />
     * %<br />
     * O0006<br />
     * G90G10L2P1<br />
     * M30<br />
     * %
     * @param program 程序内容
     * @param everyWriteSize 分割写入的长度信息
     * @return 是否下载成功
     */
    public OperateResult WriteProgramContent(String program, int everyWriteSize){
        return WriteProgramContent(program, everyWriteSize, "");
    }
    /**
     * <b>[商业授权]</b> 将指定文件的NC加工程序，下载到数控机床里，返回是否下载成功<br />
     * <b>[Authorization]</b> Download the NC machining program of the specified file to the CNC machine tool, and return whether the download is successful<br />
     * <br />
     * 程序文件的内容必须%开始，%结束，下面是一个非常简单的例子：<br />
     * %<br />
     * O0006<br />
     * G90G10L2P1<br />
     * M30<br />
     * %
     * @param program 程序内容
     * @param everyWriteSize 分割写入的长度信息
     * @param path 程序路径信息，默认为空，就是 //CNC_MEM/USER/PATH1/ 如果需要指定PATH2，需要输入 //CNC_MEM/USER/PATH2/
     * @return 是否下载成功
     */
    public OperateResult WriteProgramContent(String program, int everyWriteSize, String path) {
        if (!Authorization.asdniasnfaksndiqwhawfskhfaiw())
            return new OperateResult(StringResources.Language.InsufficientPrivileges());

        OperateResultExOne<Socket> socket = CreateSocketAndConnect(getIpAddress(), getPort(), getConnectTimeOut());
        if (!socket.IsSuccess) return socket;

        OperateResultExOne<byte[]> ini1 = ReadFromCoreServer(socket.Content, SoftBasic.HexStringToBytes("a0 a0 a0 a0 00 01 01 01 00 02 00 01"), true, true);
        if (!ini1.IsSuccess) return ini1;

        OperateResultExOne<byte[]> read1 = ReadFromCoreServer(socket.Content, BulidWriteProgramFilePre(path), true, true);
        if (!read1.IsSuccess) return read1;

        ArrayList<byte[]> contents = BulidWriteProgram(program.getBytes(StandardCharsets.US_ASCII), everyWriteSize);
        for (int i = 0; i < contents.size(); i++) {
            OperateResultExOne<byte[]> read2 = ReadFromCoreServer(socket.Content, contents.get(i), false, true);
            if (!read2.IsSuccess) return read2;
        }

        OperateResultExOne<byte[]> read3 = ReadFromCoreServer(socket.Content, new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x13, 0x01, 0x00, 0x00}, true, true);
        if (!read3.IsSuccess) return read3;

        CloseSocket(socket.Content);
        if (read3.Content.length >= 14) {
            int err = this.getByteTransform().TransInt16(read3.Content, 12);
            if (err != 0) return new OperateResultExOne<String>(err, StringResources.Language.UnknownError());
        }

        return OperateResult.CreateSuccessResult();
    }

    /**
     * <b>[商业授权]</b> 读取指定程序号的程序内容<br />
     * <b>[Authorization]</b> Read the program content of the specified program number
     * @param program 程序号
     * @return 程序内容
     */
    public OperateResultExOne<String> ReadProgram(int program){
        return ReadProgram(program, "");
    }
    /**
     * <b>[商业授权]</b> 根据路径信息读取指定程序号的程序内容<br />
     * <b>[Authorization]</b> Read the program content of the specified program number
     * @param program 程序号
     * @param path 程序路径信息，默认为空，就是 //CNC_MEM/USER/PATH1/ 如果需要指定PATH2，需要输入 //CNC_MEM/USER/PATH2/
     * @return 程序内容
     */
    public OperateResultExOne<String> ReadProgram(int program, String path) {
        if (!Authorization.asdniasnfaksndiqwhawfskhfaiw())
            return new OperateResultExOne<String>(StringResources.Language.InsufficientPrivileges());

        OperateResultExOne<Socket> socket = CreateSocketAndConnect(getIpAddress(), getPort(), getConnectTimeOut());
        if (!socket.IsSuccess) return OperateResultExOne.CreateFailedResult(socket);

        OperateResultExOne<byte[]> ini1 = ReadFromCoreServer(socket.Content, SoftBasic.HexStringToBytes("a0 a0 a0 a0 00 01 01 01 00 02 00 01"), true, true);
        if (!ini1.IsSuccess) return OperateResultExOne.CreateFailedResult(ini1);

        OperateResultExOne<byte[]> read1 = ReadFromCoreServer(socket.Content, BuildReadProgramPre(program, path), true, true);
        if (!read1.IsSuccess) return OperateResultExOne.CreateFailedResult(read1);

        // 检测错误信息
        int err = read1.Content[12] * 256 + read1.Content[13];
        if (err != 0) {
            CloseSocket(socket.Content);
            return new OperateResultExOne<String>(err, StringResources.Language.UnknownError());
        }

        StringBuilder sb = new StringBuilder();
        while (true) {
            OperateResultExOne<byte[]> read2 = ReadFromCoreServer(socket.Content, null, true, true);
            if (!read2.IsSuccess) return OperateResultExOne.CreateFailedResult(read2);

            if (read2.Content[6] == 0x16)
                sb.append(new String(read2.Content, 10, read2.Content.length - 10, StandardCharsets.US_ASCII));
            else if (read2.Content[6] == 0x17)
                break;
        }

        OperateResult send = Send(socket.Content, new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x17, 0x02, 0x00, 0x00});
        if (!send.IsSuccess) return OperateResultExOne.CreateFailedResult(send);

        CloseSocket(socket.Content);
        return OperateResultExOne.CreateSuccessResult(sb.toString());
    }

    /**
     * 根据指定的程序号信息，删除当前的程序信息<br />
     * According to the designated program number information, delete the current program information
     * @param program 程序号
     * @return 是否删除成功
     */
    public OperateResult DeleteProgram(int program) {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(
                BuildReadArray(BuildReadSingle((short) 0x05, program, 0, 0, 0, 0)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        byte[] result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10)).get(0);
        short err = this.getByteTransform().TransInt16(result, 6);

        if (err == 0) return OperateResult.CreateSuccessResult();
        else return new OperateResult(err, StringResources.Language.UnknownError());
    }

    /**
     * 读取当前程序的前台路径<br />
     * Read the foreground path of the current program
     * @return 程序的路径信息
     */
    public OperateResultExOne<String> ReadCurrentForegroundDir() {
        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(BuildReadSingle((short) 0xB0, 1, 0, 0, 0, 0)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int index = 0;
        for (int i = 14; i < result.get(0).length; i++) {
            if (result.get(0)[i] == 0x00) {
                index = i;
                break;
            }
        }
        if (index == 0) index = result.get(0).length;
        return OperateResultExOne.CreateSuccessResult(new String(result.get(0), 14, index - 14, StandardCharsets.US_ASCII));
    }

    /**
     * 设置指定路径为当前路径<br />
     * Set the specified path as the current path
     * @param programName 程序名
     * @return 结果信息
     */
    public OperateResult SetDeviceProgsCurr(String programName) {
        OperateResultExOne<String> path = ReadCurrentForegroundDir();
        if (!path.IsSuccess) return path;

        byte[] buffer = new byte[256];
        byte[] b = (path.Content + programName).getBytes(StandardCharsets.US_ASCII);
        System.arraycopy(b, 0, buffer, 0, b.length);

        OperateResultExOne<byte[]> read = ReadFromCoreServer(BuildReadArray(BuildWriteSingle((short) 0xBA, 0, 0, 0, 0, buffer)));
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        ArrayList<byte[]> result = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(read.Content, 10));
        int status = result.get(0)[10] * 256 + result.get(0)[11];

        if (status == 0) return OperateResult.CreateSuccessResult();
        else return new OperateResult(status, StringResources.Language.UnknownError());
    }

    /**
     * 读取机床的当前时间信息<br />
     * Read the current time information of the machine tool
     * @return 时间信息
     */
    public OperateResultExOne<Date> ReadCurrentDateTime() {
        OperateResultExOne<Double> read1 = ReadSystemMacroValue(3011);
        if (!read1.IsSuccess) return OperateResultExOne.CreateFailedResult(read1);

        OperateResultExOne<Double> read2 = ReadSystemMacroValue(3012);
        if (!read2.IsSuccess) return OperateResultExOne.CreateFailedResult(read2);

        String date = String.valueOf(read1.Content.intValue());
        String time = String.format("%06d", read2.Content.intValue());

        return OperateResultExOne.CreateSuccessResult(new Date(
                Integer.parseInt(date.substring(0, 4) )- 1900, Integer.parseInt(date.substring(4, 6)) - 1, Integer.parseInt(date.substring(6)),
                Integer.parseInt(time.substring(0, 2)), Integer.parseInt(time.substring(2, 4)), Integer.parseInt(time.substring(4))));
    }

    /**
     * 读取当前的已加工的零件数量<br />
     * Read the current number of processed parts
     * @return 已经加工的零件数量
     */
    public OperateResultExOne<Integer> ReadCurrentProduceCount() {
        OperateResultExOne<Double> read = ReadSystemMacroValue(3901);
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        return OperateResultExOne.CreateSuccessResult(read.Content.intValue());
    }

    /**
     * 读取期望的加工的零件数量<br />
     * Read the expected number of processed parts
     * @return 期望的加工的零件数量
     */
    public OperateResultExOne<Integer> ReadExpectProduceCount() {
        OperateResultExOne<Double> read = ReadSystemMacroValue(3902);
        if (!read.IsSuccess) return OperateResultExOne.CreateFailedResult(read);

        return OperateResultExOne.CreateSuccessResult(read.Content.intValue());
    }

    /**
     * 构建读取一个命令的数据内容
     * @param code 命令码
     * @param a 第一个参数内容
     * @param b 第二个参数内容
     * @param c 第三个参数内容
     * @param d 第四个参数内容
     * @param e 第五个参数内容
     * @return 总报文信息
     */
    private byte[] BuildReadSingle(short code, int a, int b, int c, int d, int e) {
        return BuildReadMulti((short) 0x01, code, a, b, c, d, e);
    }

    /**
     * 构建读取多个命令的数据内容
     * @param mode 模式
     * @param code 命令码
     * @param a 第一个参数内容
     * @param b 第二个参数内容
     * @param c 第三个参数内容
     * @param d 第四个参数内容
     * @param e 第五个参数内容
     * @return 总报文信息
     */
    private byte[] BuildReadMulti(short mode, short code, int a, int b, int c, int d, int e) {
        byte[] buffer = new byte[28];
        buffer[1] = 0x1C;
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(mode), buffer, 2);
        buffer[5] = 0x01;
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(code), buffer, 6);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(a), buffer, 8);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(b), buffer, 12);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(c), buffer, 16);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(d), buffer, 20);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(e), buffer, 24);
        return buffer;
    }

    /**
     * 创建写入byte[]数组的报文信息
     * @param code 命令码
     * @param a 第一个参数内容
     * @param b 第二个参数内容
     * @param c 第三个参数内容
     * @param d 第四个参数内容
     * @param data 等待写入的byte数组信息
     * @return 总报文信息
     */
    private byte[] BuildWriteSingle(short code, int a, int b, int c, int d, byte[] data) {
        return BuildWriteSingle((short) 0x01, code, a, b, c, d, data);
    }

    private byte[] BuildWriteSingle(short mode, short code, int a, int b, int c, int d, byte[] data) {
        byte[] buffer = new byte[28 + data.length];
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte((short) buffer.length), buffer, 0);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(mode), buffer, 2);
        buffer[5] = 0x01;
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(code), buffer, 6);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(a), buffer, 8);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(b), buffer, 12);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(c), buffer, 16);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(d), buffer, 20);
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte(data.length), buffer, 24);
        if (data.length > 0) Utilities.ByteArrayCopyTo(data, buffer, 28);
        return buffer;
    }
    /**
     * 创建写入单个double数组的报文信息
     * @param code 功能码
     * @param a 第一个参数内容
     * @param b 第二个参数内容
     * @param c 第三个参数内容
     * @param d 第四个参数内容
     * @param data 等待写入的double数组信息
     * @return 总报文信息
     */
    private byte[] BuildWriteSingle(short code, int a, int b, int c, int d, double[] data) {
        byte[] buffer = new byte[data.length * 8];
        for (int i = 0; i < data.length; i++) {
            Utilities.ByteArrayCopyTo(CreateFromFanucDouble(data[i]), buffer, 0);
        }
        return BuildWriteSingle(code, a, b, c, d, buffer);
    }

    /**
     * 创建多个命令报文的总报文信息
     * @param commands 报文命令的数组
     * @return 总报文信息
     */
    private byte[] BuildReadArray(byte[]... commands) {
        ByteArrayOutputStream ms = new ByteArrayOutputStream();
        ms.write(new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x21, 0x01, 0x00, 0x1e}, 0, 10);
        ms.write(getByteTransform().TransByte((short) commands.length), 0, 2);
        for (int i = 0; i < commands.length; i++) {
            ms.write(commands[i], 0, commands[i].length);
        }
        byte[] buffer = ms.toByteArray();
        Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte((short) (buffer.length - 10)), buffer, 8);
        return buffer;
    }

    private byte[] BulidWriteProgramFilePre(String path) {
        if (!Utilities.IsStringNullOrEmpty(path)) {
            if (!path.endsWith("/")) path += "/";
            if (!path.startsWith("N:")) path = "N:" + path;
        }
        ByteArrayOutputStream ms = new ByteArrayOutputStream();
        ms.write(new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x11, 0x01, 0x02, 0x04}, 0, 10);
        ms.write(new byte[]{0x00, 0x00, 0x00, 0x01}, 0, 4);
        for (int i = 0; i < 512; i++) {
            ms.write(0x00);
        }
        byte[] buffer = ms.toByteArray();
        if (!Utilities.IsStringNullOrEmpty(path)){
            Utilities.ByteArrayCopyTo(path.getBytes(StandardCharsets.US_ASCII), buffer, 14);
        }
        return buffer;
    }

    /**
     * 创建读取运行程序的报文信息
     * @param program 程序号
     * @param path 程序的路径
     * @return 总报文
     */
    private byte[] BuildReadProgramPre(int program, String path) {
        if (!Utilities.IsStringNullOrEmpty(path)) {
            if (!path.endsWith("/")) path += "/";
            if (!path.startsWith("N:")) path = "N:" + path;
        }

        ByteArrayOutputStream ms = new ByteArrayOutputStream();
        ms.write(new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x15, 0x01, 0x02, 0x04}, 0, 10);
        ms.write(new byte[]{0x00, 0x00, 0x00, 0x01}, 0, 4);
        for (int i = 0; i < 512; i++) {
            ms.write(0x00);
        }
        byte[] buffer = ms.toByteArray();
        String pro = Utilities.IsStringNullOrEmpty(path) ? "O" + program + "-" + "O" + program : path + "O" + program;
        Utilities.ByteArrayCopyTo(pro.getBytes(StandardCharsets.US_ASCII), buffer, 14);
        return buffer;
    }

    private ArrayList<byte[]> BulidWriteProgram(byte[] program, int everyWriteSize) {
        ArrayList<byte[]> list = new ArrayList<>();
        int[] lengths = SoftBasic.SplitIntegerToArray(program.length, everyWriteSize);
        int index = 0;
        for (int i = 0; i < lengths.length; i++) {
            ByteArrayOutputStream ms = new ByteArrayOutputStream();
            ms.write(new byte[]{(byte) 0xa0, (byte) 0xa0, (byte) 0xa0, (byte) 0xa0, 0x00, 0x01, 0x12, 0x04, 0x00, 0x00}, 0, 10);
            ms.write(program, index, lengths[i]);
            byte[] buffer = ms.toByteArray();
            Utilities.ByteArrayCopyTo(this.getByteTransform().TransByte((short) (buffer.length - 10)), buffer, 8);

            list.add(buffer);
            index += lengths[i];
        }
        return list;
    }

    /**
     * 从机床返回的数据里解析出实际的数据内容，去除了一些多余的信息报文。
     * @param content 返回的报文信息
     * @return 解析之后的报文信息
     */
    private ArrayList<byte[]> ExtraContentArray(byte[] content) {
        ArrayList<byte[]> list = new ArrayList<>();
        int count = getByteTransform().TransUInt16(content, 0);
        int index = 2;

        for (int i = 0; i < count; i++) {
            int length = getByteTransform().TransUInt16(content, index);
            list.add(SoftBasic.BytesArraySelectMiddle(content, index + 2, length - 2));
            index += length;
        }
        return list;
    }

    private OperateResultExOne<CutterInfo[]> ExtraCutterInfos(byte[] content1, byte[] content2, byte[] content3, byte[] content4, int cutterNumber) {
        // 先提取出各个数据信息
        ArrayList<byte[]> result1 = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(content1, 10));
        ArrayList<byte[]> result2 = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(content2, 10));
        ArrayList<byte[]> result3 = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(content3, 10));
        ArrayList<byte[]> result4 = ExtraContentArray(SoftBasic.BytesArrayRemoveBegin(content4, 10));

        // 校验数据是否有效
        boolean check1 = this.getByteTransform().TransInt16(result1.get(0), 6) == 0x00;
        boolean check2 = this.getByteTransform().TransInt16(result2.get(0), 6) == 0x00;
        boolean check3 = this.getByteTransform().TransInt16(result3.get(0), 6) == 0x00;
        boolean check4 = this.getByteTransform().TransInt16(result4.get(0), 6) == 0x00;

        // 如果数据有效，则显示出来
        CutterInfo[] cutters = new CutterInfo[cutterNumber];
        for (int i = 0; i < cutters.length; i++) {
            cutters[i] = new CutterInfo();
            cutters[i].LengthSharpOffset = check1 ? GetFanucDouble(result1.get(0), 14 + 8 * i) : Double.NaN;
            cutters[i].LengthWearOffset  = check2 ? GetFanucDouble(result2.get(0), 14 + 8 * i) : Double.NaN;
            cutters[i].RadiusSharpOffset = check3 ? GetFanucDouble(result3.get(0), 14 + 8 * i) : Double.NaN;
            cutters[i].RadiusWearOffset  = check4 ? GetFanucDouble(result4.get(0), 14 + 8 * i) : Double.NaN;
        }

        return OperateResultExOne.CreateSuccessResult(cutters);
    }

    private Charset encoding;
    private FanucSysInfo fanucSysInfo;

    public String toString() {
        return "FanucSeries0i[" + getIpAddress() + ":" + getPort() + "]";
    }
}
