package com.netfinworks.mq;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import javax.jms.ConnectionFactory;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.pool.PooledConnectionFactory;
import org.springframework.jms.remoting.JmsInvokerProxyFactoryBean;

import com.netfinworks.mq.core.MQService;
import com.netfinworks.mq.jms.DestinationResolver;
import com.netfinworks.mq.jms.impl.DestinationInfoFactory;
import com.netfinworks.mq.jms.impl.DestinationResolverImpl;
import com.netfinworks.mq.jms.impl.JmsAccessorImpl;
import com.netfinworks.mq.jms.impl.JmsService;
import com.netfinworks.mq.jms.impl.JmsTemplate;
import com.netfinworks.mq.management.service.DestinationInfoService;

/**
 * 创建MQService的实例，保证相同上下文资源只有一个MQService实例，以减少系统资源开销
 *
 */
public class MQServiceCreator {
	private static Object monitor = new Object();
	private static final String DESTINATION_INFO_QUEUE = "netfinworks.amo.destination";
	private static Map<Hashtable<String, String>, MQService> mqServiceMap = new HashMap<Hashtable<String, String>, MQService>();
	private static MQService mqService = null;
	
	private MQServiceCreator() {
	}
	
	/**
	 * 创建MQService的实例（通过netfinworks-mq.properties配置文件创建Jms连接）
	 * @return MQService的实例
	 */
	public static MQService getMQService(){
		if (mqService == null) {
			synchronized (monitor) {
				if (mqService == null) {
					mqService = createMQService(null);
				}
			}
		}

		return mqService;
	}

	/**
	 * 创建MQService的实例
	 * @param environment	创建Jms连接的JNDI上下文
	 * @return MQService的实例
	 */
	public static MQService getMQService(Hashtable<String, String> environment) {
		MQService mqService = mqServiceMap.get(environment);

		if (mqService == null) {
			synchronized (monitor) {
				mqService = mqServiceMap.get(environment);

				if (mqService == null) {
					mqService = createMQService(environment);
					
					mqServiceMap.put(copyHashtable(environment), mqService);
				}
			}
		}

		return mqService;
	}

	/**
	 * 创建MQService
	 * @param environment
	 * @return
	 */
	private static MQService createMQService(Hashtable<String, String> environment) {
		ConnectionFactory connectionFactory = createConnectionFactory(environment);
		DestinationResolver destinationResolver = DestinationResolverImpl.getInstance();		
		
		JmsTemplate jmsTemplate = new JmsTemplate();
		jmsTemplate.setConnectionFactory(connectionFactory);
		jmsTemplate.setDestinationResolver(destinationResolver);
		
		JmsAccessorImpl jmsAccessor = new JmsAccessorImpl();
		jmsAccessor.setJmsTemplate(jmsTemplate);
		jmsAccessor.setDestinationInfoFactory(createDestinationInfoFactory(connectionFactory));
		
		JmsService mqService = new JmsService();
		mqService.setMqAccessor(jmsAccessor);		
		
		return mqService;
	}
	
	private static ConnectionFactory createConnectionFactory(
			Hashtable<String, String> environment) {
		if(environment == null)
			throw new NullPointerException();
		
		String url = environment.get("java.naming.provider.url");
		
		if(url == null){
			throw new NullPointerException();
		}
		
		ActiveMQConnectionFactory activeMQConnectionFactory = new ActiveMQConnectionFactory(url);
		if(environment.containsKey("java.naming.security.principal")){
			activeMQConnectionFactory.setUserName(environment.get("java.naming.security.principal"));
			activeMQConnectionFactory.setPassword(environment.get("java.naming.security.credentials"));
		}
		
		return new PooledConnectionFactory(activeMQConnectionFactory);
	}

	/**
	 * 创建DestinationInfoFactory
	 */
	private static DestinationInfoFactory createDestinationInfoFactory(ConnectionFactory connectionFactory){
		DestinationInfoFactory destinationInfoFactory = new DestinationInfoFactory();
		
		JmsInvokerProxyFactoryBean proxy = new JmsInvokerProxyFactoryBean();
		proxy.setConnectionFactory(connectionFactory);
		proxy.setQueueName(DESTINATION_INFO_QUEUE);
		proxy.setServiceInterface(DestinationInfoService.class);
		proxy.afterPropertiesSet();
		
		destinationInfoFactory.setDestinationInfoService((DestinationInfoService) proxy.getObject());
		
		return destinationInfoFactory;
	}

	private static Hashtable<String, String> copyHashtable(
			Hashtable<String, String> environment) {
		Hashtable<String, String> env = new Hashtable<String, String>();
		
		for(String key : environment.keySet())
		{
			env.put(key, environment.get(key));
		}
		
		return env;
	}
}
