package com.yqbsoft.laser.api;

import java.io.IOException;
import java.security.Security;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import com.yqbsoft.laser.api.LaserApiException;
import com.yqbsoft.laser.api.LaserClient;
import com.yqbsoft.laser.api.LaserConstants;
import com.yqbsoft.laser.api.LaserParser;
import com.yqbsoft.laser.api.LaserRequest;
import com.yqbsoft.laser.api.LaserResponse;
import com.yqbsoft.laser.api.internal.parser.json.ObjectJsonParser;
import com.yqbsoft.laser.api.internal.parser.xml.ObjectXmlParser;
import com.yqbsoft.laser.api.internal.util.RequestParametersHolder;
import com.yqbsoft.laser.api.internal.util.StringUtils;
import com.yqbsoft.laser.api.internal.util.LaserHashMap;
import com.yqbsoft.laser.api.internal.util.LaserLogger;
import com.yqbsoft.laser.api.internal.util.LaserUtils;
import com.yqbsoft.laser.api.internal.util.WebUtils;

public class DefaultInnerClient implements LaserClient {

    private String serverUrl;
    private String appId;
//    private String privateKey; // 渠道方私钥
//    private String publicKey; // Open平台公钥
    private String format         = LaserConstants.FORMAT_JSON;
//    private String sign_type      = LaserConstants.SIGN_TYPE_RSA;
    private String charset        = LaserConstants.CHARSET_UTF8;

    private int    connectTimeout = 3000;
    private int    readTimeout    = 15000;

    static {
        //清除安全设置
        Security.setProperty("jdk.certpath.disabledAlgorithms", "");
    }

    public DefaultInnerClient(String serverUrl, String appId) {
        this(serverUrl, appId, null, null);
    }
    
    public DefaultInnerClient(String serverUrl, String appId, String format, String charset) {
        this.serverUrl = serverUrl;
        this.appId = appId;
//        this.privateKey = privateKey;
//        this.publicKey = publicKey;
        this.format = format;
        this.charset = charset;
    }

    public <T extends LaserResponse> T execute(LaserRequest<T> request) throws LaserApiException {
        return execute(request, null);
    }

    public <T extends LaserResponse> T execute(LaserRequest<T> request, String accessToken)
                                                                                                 throws LaserApiException {
        LaserParser<T> parser = null;
        if (LaserConstants.FORMAT_XML.equals(this.format)) {
            parser = new ObjectXmlParser<T>(request.getResponseClass());
        } else {
            parser = new ObjectJsonParser<T>(request.getResponseClass());
        }

        return _execute(request, parser, accessToken);
    }

    private <T extends LaserResponse> T _execute(LaserRequest<T> request, LaserParser<T> parser, String authToken)
                                                                                                                         throws LaserApiException {
        Map<String, Object> rt = doPost(request, authToken);
        if (rt == null) {
            return null;
        }
        T tRsp = null;
        try {
            tRsp = parser.parse(rt.get("rsp"));
            if(tRsp == null) return null;
        } catch (RuntimeException e) {
            LaserLogger.logBizError((String) rt.get("rsp"));
            throw e;
        } catch (LaserApiException e) {
            LaserLogger.logBizError((String) rt.get("rsp"));
            throw new LaserApiException(e);
        }
        if (!tRsp.isSuccess()) {
            LaserLogger.logErrorScene(rt, tRsp, "");
        }
        return tRsp;
    }

    public <T extends LaserResponse> Map<String, Object> doPost(LaserRequest<T> request, String accessToken)
                                                                                                                  throws LaserApiException {
        Map<String, Object> result = new HashMap<String, Object>();
        RequestParametersHolder requestHolder = new RequestParametersHolder();
        LaserHashMap appParams = new LaserHashMap(request.getTextParams());
        requestHolder.setApplicationParams(appParams);

        LaserHashMap protocalMustParams = new LaserHashMap();
        LaserHashMap protocalOptParams = new LaserHashMap();
        
        protocalMustParams.put(LaserConstants.APP_ID, this.appId);
        protocalMustParams.put(LaserConstants.METHOD, request.getApiMethodName());
        protocalMustParams.put(LaserConstants.VERSION, request.getApiVersion());
        protocalMustParams.put(LaserConstants.PROD_CODE, request.getProdCode());
        protocalMustParams.put(LaserConstants.TERMINAL_TYPE, request.getTerminalType());
        protocalMustParams.put(LaserConstants.TERMINAL_INFO, request.getTerminalInfo());

        Long timestamp = System.currentTimeMillis();
        DateFormat df = new SimpleDateFormat(LaserConstants.DATE_TIME_FORMAT);
        df.setTimeZone(TimeZone.getTimeZone(LaserConstants.DATE_TIMEZONE));
        protocalMustParams.put(LaserConstants.TIMESTAMP, df.format(new Date(timestamp)));
        requestHolder.setProtocalMustParams(protocalMustParams);

        protocalOptParams.put(LaserConstants.FORMAT, format);
        protocalOptParams.put(LaserConstants.ACCESS_TOKEN, accessToken);
        protocalOptParams.put(LaserConstants.OPEN_SDK, LaserConstants.SDK_VERSION);
        protocalOptParams.put(LaserConstants.PROD_CODE, request.getProdCode());
        requestHolder.setProtocalOptParams(protocalOptParams);

        if (StringUtils.isEmpty(charset)) {
            charset = LaserConstants.CHARSET_UTF8;
        }

        StringBuffer urlSb = new StringBuffer(serverUrl);
        try {
            String sysMustQuery = WebUtils.buildQuery(requestHolder.getProtocalMustParams(), charset);
            String sysOptQuery = WebUtils.buildQuery(requestHolder.getProtocalOptParams(), charset);

            urlSb.append("?");
            urlSb.append(sysMustQuery);
            if (sysOptQuery != null & sysOptQuery.length() > 0) {
                urlSb.append("&");
                urlSb.append(sysOptQuery);
            }
        } catch (IOException e) {
            throw new LaserApiException(e);
        }
        String rsp = null;
        try {
            rsp = WebUtils.doPost(urlSb.toString(), appParams, charset, connectTimeout, readTimeout);
        } catch (IOException e) {
            throw new LaserApiException(e);
        }
        
        //result.put(LaserConstants.SIGN, LaserUtils.getSignFromResponseJson(rsp));
       // result.put(LaserConstants.RS_TIMESTAMP, LaserUtils.getTimestampFromResponseJson(rsp));
        result.put(LaserConstants.SIGN_CONTENT, LaserUtils.getSignContentFromResponseJson(rsp));
       // result.put("rsp", rsp);
       // result.put("textParams", appParams);
       // result.put("protocalMustParams", protocalMustParams);
       // result.put("protocalOptParams", protocalOptParams);
       // result.put("url", urlSb.toString());
        return result;
    }

    
    /**
     * 获取平台授权Token的步骤.
     * 1. 浏览器重定向到CodeRequestURL的请求，向平台发起请求。
     * 2. 用户在平台登录、授权，平台向渠道分发授权code(通过redirect_uri后面挂参数重定向传递)。
     * 3. 渠道通过redirect_uri请求接收到参数信息(主要是code, expires_in)。
     * 4. 渠道用code组成TokenRequestURL从后台向平台交换有效Token(username, token, refresh_token, expires_in)
     * 5. 渠道通过平台API访问需要用户授权信息时，携带用户相关Token即可
     */
    
    /**
     * 生成OAuth2的Code请求URL
     * @throws LaserApiException 
     */
    @Override
    public String generateCodeRequestURL(String oauthServer, String redirectUrl, HashMap<String, Object> params) throws LaserApiException {
        if(StringUtils.isEmpty(oauthServer) || StringUtils.isEmpty(redirectUrl)){
            throw new LaserApiException("oauthServer and redirectUrl is not be null.");
        }
        if(params == null) params = new HashMap<String, Object>();
        params.put(LaserConstants.OAUTH2_RESPONSE_TYPE, LaserConstants.OAUTH2_TYPE_CODE);
        params.put(LaserConstants.OAUTH2_REDIRECT_URI, redirectUrl);
        processRequireParamAndSign(params);
        return oauthServer + "?" + StringUtils.formatToURLParam(params.entrySet(), LaserConstants.CHARSET_UTF8);
    }
    
    /**
     * 解析OAuth2的Code响应URL
     * @throws LaserApiException 
     */
    @Override
    public boolean checkOauthResponseSign(Map<String, String[]> params) throws LaserApiException {
        return true;
    }
    
    /**
     * 生成OAuth2的Token请求URL.
     * @param oauthServer
     * @param redirectUrl
     * @param code
     * @param params
     * @return
     * @throws LaserApiException
     */
    @Override
    public String generateTokenRequestURL(String oauthServer, String redirectUrl, String code, HashMap<String, Object> params) throws LaserApiException {
        if(StringUtils.isEmpty(oauthServer) || StringUtils.isEmpty(redirectUrl)){
            throw new LaserApiException("oauthServer and redirectUrl is not be null.");
        }
        if(params == null) params = new HashMap<String, Object>();
        params.put(LaserConstants.OAUTH2_GRANT_TYPE_KEY, LaserConstants.OAUTH2_ACCESS_TOKEN);
        params.put(LaserConstants.OAUTH2_TYPE_CODE, code);
        params.put(LaserConstants.OAUTH2_REDIRECT_URI, redirectUrl);
        processRequireParamAndSign(params);
        return oauthServer + "?" + StringUtils.formatToURLParam(params.entrySet(), LaserConstants.CHARSET_UTF8);
    }
    /**
     * 生成OAuth2的RefreshToken请求URL.
     * @param oauthServer
     * @param redirectUrl
     * @param code
     * @param params
     * @return
     * @throws LaserApiException
     */
    public String generateRefreshTokenRequestURL(String oauthServer, String redirectUrl, String refreshToken, HashMap<String, Object> params) throws LaserApiException {
        if(StringUtils.isEmpty(oauthServer) || StringUtils.isEmpty(redirectUrl)){
            throw new LaserApiException("oauthServer and redirectUrl is not be null.");
        }
        if(params == null) params = new HashMap<String, Object>();
        params.put(LaserConstants.OAUTH2_GRANT_TYPE_KEY, LaserConstants.OAUTH2_REFRESH_TOKEN);
        params.put(LaserConstants.OAUTH2_TYPE_CODE, refreshToken);
        params.put(LaserConstants.OAUTH2_REDIRECT_URI, redirectUrl);
        processRequireParamAndSign(params);
        return oauthServer + "?" + StringUtils.formatToURLParam(params.entrySet(), LaserConstants.CHARSET_UTF8);
    }
    
    /**
     * 处理appKey，randomCode和签名.
     * 内部Client不需要处理签名问题
     * @param params
     * @return
     * @throws LaserApiException
     */
    private Map<String, Object> processRequireParamAndSign(Map<String, Object> params) throws LaserApiException{
        if(params == null) params = new HashMap<String, Object>();
        params.put(LaserConstants.OAUTH2_APP_KEY, appId);
        params.put(LaserConstants.OAUTH2_RANDOM_CODE, new Date().getTime()+"");
        return params;
    }
}
