/*
 * Copyright (c) 2002-2018 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This product is licensed to you under the Apache License, Version 2.0 (the "License").
 * You may not use this product except in compliance with the License.
 *
 * This product may include a number of subcomponents with
 * separate copyright notices and license terms. Your use of the source
 * code for these subcomponents is subject to the terms and
 *  conditions of the subcomponent's license, as noted in the LICENSE file.
 */
package org.neo4j.ogm.spi;

import java.util.Map;
import java.util.function.Function;

/**
 * This interface can be implemented by extensions that wish to modify Cypher statements before they are being sent
 * by corresponding OGM driver to the database.<br>
 * <br>
 * OGM makes sure that the modification will be applied to all statements, regardless of source, but implementations of
 * the service must not rely on the fact that the modification happens currently on the drivers level, which may change
 * without notice.<br>
 * <br>
 * To add your custom modification, create a separate module, implement this interface and return an arbitrary
 * {@link Function<String, String>}. As an optional step, overwrite the {@link #getOrder()} method in case you have
 * multiple modifications on the class path. The lower the value it returns, the higher it's priority. Higher priority
 * means modification is applied first.<br>
 * <br>
 * OGM loads implementations through Java's builtin Service Provider Interfache mechanism.
 * Please refer to <a href="https://docs.oracle.com/javase/tutorial/sound/SPI-intro.html">Oracle's</a>
 * documentation of Java's Service Provide Interface to declare your implementation. In short: Define a file called
 * <code>META-INF/services/org.neo4j.ogm.spi.CypherModificationProvider</code> in your module containing the fully qualified
 * name of your <code>CypherModificationProvider</code>-Implementation.<br>
 * <br>
 * Inside the function returned by {@link #getCypherModification(Map)}} the Cypher modification itself is supposed to happend.
 * The function is called with the Cypher string generated by OGM or passed to the Session interface and is then supposed
 * to return a non null, non empty string.<br>
 * <br>
 * The map passed to {@link #getCypherModification(Map)}} contains arbitrary properties from OGMs
 * {@link org.neo4j.ogm.config.Configuration}. Those can be set programmatically in any way necessary. Use those properties
 * to configure your Cypher modification as needed.
 *
 * @author Michael J. Simons
 * @since 3.1.4
 */
public interface CypherModificationProvider {
    /**
     * Get the order value of this object.
     * Higher values are interpreted as lower priority. As a consequence, the object with the lowest value has the highest priority.
     * Same order values will result in arbitrary sort positions for the affected objects.
     *
     * @return the order value
     */
    default int getOrder() {
        return Integer.MAX_VALUE;
    }

    /**
     * Called by OGM to retrieve the actual Cypher modification.
     *
     * @param configuration Map containing all custom properties from {@link org.neo4j.ogm.config.Configuration}
     * @return The actual cypher modification
     */
    Function<String, String> getCypherModification(Map<String, Object> configuration);
}
